#version 420

in vec4 model_Vertex;
in vec3 view_Vertex;
in vec3 view_Normal;
in vec3 view_Tangent;
in vec3 view_Binormal;
in vec3 view_Light_Positions[3];
in vec3 world_Normal;

in vec2 texture_Coord;

uniform sampler2D texture_Ambient;
uniform sampler2D texture_Diffuse;
uniform sampler2D texture_Normal;

uniform sampler3D light_DepthMaps[3];
uniform sampler3D light_ColorMaps[3];
uniform sampler3D light_NormalMaps[3];

uniform mat4 light_ModelViewMatrices[3];
uniform vec4 light_Colors[3];
uniform float light_FarPlane;

uniform vec3 volume_Min;
uniform vec3 volume_Max;
uniform int volume_Size;

uniform sampler3D volume_LightPropagation[3];
uniform sampler3D volume_Geometry;

/*uniform int config_lpvPropagationSteps;
uniform float config_lpvPropagationFactor;
uniform float config_lpvInjectionFactor;*/
uniform float config_lpvRenderFactor;
uniform bool config_useDirectLightning;
uniform bool config_useIndirectLightning;
/* uniform bool config_useDynamicLpvs; */
uniform bool config_useDiffuseMaps;
uniform bool config_useBumpMaps;
uniform bool config_useShadowMaps;
/*uniform bool config_useIndirectOcclusion;*/

out vec4 fragmentColor;

#define EXTRACT_DEPTH(cc)	((cc).b + (cc).g / 256.0 + (cc).r / (256.0 * 256.0) + (cc).a / (256.0 * 256.0 * 256.0))
#define PI			(3.1415)

vec3 mapCube3D(in vec3 d) {
	int face = 0;
	float s = 0.0;
	float t = 0.0;
	
	vec3 absd = abs(d);
	vec3 signd = sign(d);
        float sc = 0.0;
        float tc = 0.0;
        float ma = 0.0;

        if (absd.x >= absd.y && absd.x >= absd.z) {
                sc = signd.x * d.z;
                tc = d.y;
                ma = absd.x;
                face = d.x > 0.0 ? 0 : 1;
        }

        if (absd.y >= absd.x && absd.y >= absd.z) {
                sc = -d.x;
                tc = -signd.y * d.z;
                ma = absd.y;
                face = d.y > 0.0 ? 2 : 3;
        }

        if (absd.z >= absd.x && absd.z >= absd.y) {
                sc = -signd.z * d.x;
                tc = d.y;
                ma = absd.z;
                face = d.z > 0.0 ? 4 : 5;
        }

        if (ma == 0.0) {
                s = 0.0;
                t = 0.0;
        } else {
                s = 0.5 * (sc / ma + 1.0);
                t = 0.5 * (tc / ma + 1.0);
        }
        
	return vec3(s, t, face/5.0);
}

vec4 constructSHClampedCosineLobeAroundDirection(in vec3 direction) {
	/*const float sqrtPi = sqrt(PI);
	const float sqrtPiOver2 = 0.5 * sqrtPi;
	const float sqrtPiOverSqrt3 = sqrtPi / sqrt(3.0);
	
	return vec4(sqrtPiOver2, -sqrtPiOverSqrt3*direction.y, sqrtPiOverSqrt3*direction.z, -sqrtPiOverSqrt3*direction.x);*/
	/*return vec4(
		PI * 0.282094792,
		((2.0*PI)/3.0) * -0.4886025119 * direction.y, 
		((2.0*PI)/3.0) *  0.4886025119 * direction.z, 
		((2.0*PI)/3.0) * -0.4886025119 * direction.x
	);*/
	
	return vec4(
			0.5, -0.75 * direction.y, 0.75 * direction.z, -0.75 * direction.x
		);
}

vec3 lookupVolume(in vec3 pos, in vec3 normal) {
	
	vec3 gridSize = volume_Max - volume_Min;
	vec3 tc = (pos - volume_Min) / gridSize;
	
	
	if(tc.x < 0.0 || tc.x > 1.0 ||
			tc.y < 0.0 || tc.y > 1.0 ||
			tc.z < 0.0 || tc.z > 1.0) {
		return vec3(0.0, 0.0, 0.0);
	} else {
		vec4 surfSH = constructSHClampedCosineLobeAroundDirection(normal);
		
		vec4 lpvRed = texture(volume_LightPropagation[0], tc);
		vec4 lpvGreen = texture(volume_LightPropagation[1], tc);
		vec4 lpvBlue = texture(volume_LightPropagation[2], tc);
		
		vec3 color = vec3(
				max(0.0, dot(surfSH, lpvRed)), 
				max(0.0, dot(surfSH, lpvGreen)), 
				max(0.0, dot(surfSH, lpvBlue))
			);
		//color = texture(volume_Geometry, tc).rgb;
		return color;
	}
}

void main() {
	
	vec3 N = view_Normal;
	if(config_useBumpMaps) {
		// Perform bump mapping
		mat3 TBN = mat3(normalize(view_Tangent), normalize(view_Binormal), normalize(view_Normal));
	
		vec3 bumpNormal = texture2D(texture_Normal, texture_Coord).xyz;
		if(bumpNormal == 0.0) {
			bumpNormal = vec3(0.5,0.5,1.0);
		}
		bumpNormal = 2.0 * bumpNormal - 1.0;
	
		N = normalize(TBN * bumpNormal);
	}
	
	vec3 E = normalize(-view_Vertex);
	
	vec3 albedo = vec3(1.0, 1.0, 1.0);
	
	if(config_useDiffuseMaps) {
		albedo = texture(texture_Diffuse, texture_Coord).xyz;
	}
	
	// Calcualte direct lightning contribution
	vec4 diffuse = vec4(0.0, 0.0, 0.0, 0.0);
	vec4 specular = vec4(0.0, 0.0, 0.0, 0.0);
	vec4 ambient = vec4(albedo * vec3(0.2, 0.2, 0.2), 1.0);
	vec4 indirect = vec4(0.0, 0.0, 0.0, 0.0);
	
	
	if(config_useDirectLightning) {
		for(int i = 0; i < 3; i++) {
		
			vec4 lv = light_ModelViewMatrices[i] * model_Vertex;
			vec3 v = lv.xyz/lv.w;
		
			float shadowFactor;
		
			if(config_useShadowMaps) {
				ivec3 sz = textureSize(light_DepthMaps[i], 0);
				vec3 res = vec3(1.0 / sz.xy, 0.0);
				shadowFactor = 0.0;
				vec3 tc = mapCube3D(v);
				
				float lv_CurrentDepth = length(lv.xyz / lv.w)/light_FarPlane;
				
				for(float ix = -0.5; ix <= 0.5; ix += 1.0) {
					for(float iy = -0.5; iy <= 0.5; iy += 1.0) {
						vec4 cubeMapData = texture(light_DepthMaps[i], tc + res * vec3(ix, iy, 0.0));
						float lv_StoredDepth = EXTRACT_DEPTH(cubeMapData);
						if(lv_CurrentDepth < lv_StoredDepth + 0.005) {
							shadowFactor += 1.0;
						}
					}
				}
				shadowFactor /= 4.0;
			} else {
				shadowFactor = 1.0;
			}
		
			vec3 L = normalize(view_Light_Positions[i]-view_Vertex);
			vec3 R = normalize(-reflect(L,N));
			diffuse += vec4(shadowFactor * light_Colors[i].xyz * 0.8 * max(dot(N,L), 0.0) * albedo, 0.0);
			specular += vec4(shadowFactor * light_Colors[i].xyz * 0.2 * pow(max(dot(R,E), 0.0), 5.0), 0.0);
		}
		diffuse += vec4(0.0, 0.0, 0.0, 1.0);
		specular += vec4(0.0, 0.0, 0.0, 1.0);
	}
	
	// Calculate indirect lightning contribution
	if(config_useIndirectLightning) {
		indirect = vec4(config_lpvRenderFactor * lookupVolume(model_Vertex.xyz / model_Vertex.w, -normalize(world_Normal)) * albedo, 1.0);
	}
	
	vec4 color = ambient + diffuse + specular + indirect;
	fragmentColor = vec4(color.xyz / color.w, 1.0);
}

