#include "CallbackUtils.h"
#include "MathUtils.h"
#include "Defines.h"
#include "TimeMonitor.h"

#include <osg/LightSource>
#include <osg/Light>
#include <osg/Vec3>
#include <osg/Vec4>
#include <osg/Matrixd>
#include <osg/TextureCubeMap>
#include <osgUtil/CullVisitor>
#include <osg/MatrixTransform>

#include <iostream>
#include <cstring>
#include <cmath>

BeforePrerenderCallback::BeforePrerenderCallback() {}

void BeforePrerenderCallback::operator()(osg::RenderInfo& renderInfo) const {
	TimeMonitor::instance()->tic(TimeMonitor::RSMCreate);
}

AfterPrerenderCallback::AfterPrerenderCallback() {}

void AfterPrerenderCallback::operator()(osg::RenderInfo& renderInfo) const {
	TimeMonitor::instance()->toc(TimeMonitor::RSMCreate);
}


CopyLightPositionToUniformCallback::CopyLightPositionToUniformCallback(osg::Uniform* uniform, int i) : uniform(uniform), i(i) {}

void CopyLightPositionToUniformCallback::operator() (osg::Node *node, osg::NodeVisitor *nv) {
	osg::LightSource* ls = dynamic_cast<osg::LightSource*>(node);
	osgUtil::CullVisitor* cv = dynamic_cast<osgUtil::CullVisitor*>(nv);
	if(ls && cv) {
		osg::Matrixd mvi = *cv->getModelViewMatrix() * cv->getCurrentCamera()->getInverseViewMatrix();
		osg::Light* light = ls->getLight();
		uniform->setElement(i, light->getPosition() * mvi);
	}
	this->traverse(node, nv);
}


CopyLightColorToUniformCallback::CopyLightColorToUniformCallback(osg::Uniform* uniform, int i) : uniform(uniform), i(i) {}

void CopyLightColorToUniformCallback::operator() (osg::Node *node, osg::NodeVisitor *nv) {
	osg::LightSource* ls = dynamic_cast<osg::LightSource*>(node);
	if(ls) {
		osg::Light* light = ls->getLight();
		uniform->setElement(i, light->getDiffuse());
	}
	this->traverse(node, nv);
}

CopyLightViewToUniformCallback::CopyLightViewToUniformCallback(osg::Uniform* uniform, int i) : uniform(uniform), i(i) {}

void CopyLightViewToUniformCallback::operator() (osg::Node *node, osg::NodeVisitor *nv) {
	osg::LightSource* ls = dynamic_cast<osg::LightSource*>(node);
	osgUtil::CullVisitor* cv = dynamic_cast<osgUtil::CullVisitor*>(nv);
	if(ls && cv) {
		osg::Matrixd mvi = *cv->getModelViewMatrix() * cv->getCurrentCamera()->getInverseViewMatrix();
		osg::Light* light = ls->getLight();
		osg::Vec4 pos4 = light->getPosition() * mvi;
		osg::Vec3 pos(pos4.x(), pos4.y(), pos4.z());
		pos /= pos4.w();
		osg::Matrixd viewMatrix;
		viewMatrix.makeLookAt(pos, pos-osg::Z_AXIS, osg::Y_AXIS);
		uniform->setElement(i, viewMatrix);
	}
	this->traverse(node, nv);
}


CopyLightViewToCudaMemoryCallback::CopyLightViewToCudaMemoryCallback(Transform* transform) : transform(transform) {}

void CopyLightViewToCudaMemoryCallback::operator() (osg::Node *node, osg::NodeVisitor *nv) {
	osg::LightSource* ls = dynamic_cast<osg::LightSource*>(node);
	osgUtil::CullVisitor* cv = dynamic_cast<osgUtil::CullVisitor*>(nv);
	if(ls && cv) {
		osg::Matrixd mvi = *cv->getModelViewMatrix() * cv->getCurrentCamera()->getInverseViewMatrix();
		osg::Light* light = ls->getLight();
		osg::Vec4 pos4 = light->getPosition() * mvi;
		osg::Vec3 pos(pos4.x(), pos4.y(), pos4.z());
		pos /= pos4.w();
		osg::Matrixd viewMatrix;
		osg::Matrixd viewMatrixInverse;
		viewMatrix.makeLookAt(pos, pos-osg::Z_AXIS, osg::Y_AXIS);
		viewMatrixInverse.invert(viewMatrix);

		for(int i = 0; i < 4; i++) {
			for(int j = 0; j < 4; j++) {
				transform->v[i*4 + j] = viewMatrixInverse(j, i);
			}
		}

	}
	this->traverse(node, nv);
}


UpdateCameraCallback::UpdateCameraCallback(osg::Camera *camera, int face) : camera(camera), face(face) {}

void UpdateCameraCallback::operator() (osg::Node *node, osg::NodeVisitor *nv) {
	osg::LightSource* ls = dynamic_cast<osg::LightSource*>(node);
	osgUtil::CullVisitor* cv = dynamic_cast<osgUtil::CullVisitor*>(nv);
	if(ls && cv) {
		osg::Matrixd mvi = *cv->getModelViewMatrix() * cv->getCurrentCamera()->getInverseViewMatrix();
		osg::Light* light = ls->getLight();
		osg::Vec4 pos4 = light->getPosition() * mvi;
		osg::Vec3 pos(pos4.x(), pos4.y(), pos4.z());
		pos /= pos4.w();
		osg::Matrixd viewMatrix;
		osg::Vec3f f;
		osg::Vec3f u;
		switch(face) {
		case osg::TextureCubeMap::POSITIVE_X:
			f = osg::X_AXIS;
			u = osg::Y_AXIS;
			break;
		case osg::TextureCubeMap::POSITIVE_Y:
			f = osg::Y_AXIS;
			u = -osg::Z_AXIS;
			break;
		case osg::TextureCubeMap::POSITIVE_Z:
			f = osg::Z_AXIS;
			u = osg::Y_AXIS;
			break;
		case osg::TextureCubeMap::NEGATIVE_X:
			f = -osg::X_AXIS;
			u = osg::Y_AXIS;
			break;
		case osg::TextureCubeMap::NEGATIVE_Y:
			f = -osg::Y_AXIS;
			u = osg::Z_AXIS;
			break;
		case osg::TextureCubeMap::NEGATIVE_Z:
			f = -osg::Z_AXIS;
			u = osg::Y_AXIS;
			break;
		default:
			std::cout << "Attempted to update a camera for an invalid face!" << std::endl;
			break;
		}
		viewMatrix.makeLookAt(pos, pos+f, u);
		camera->setViewMatrix(viewMatrix);

	}
	this->traverse(node, nv);
}

FlagRemover::FlagRemover() : osg::NodeVisitor(NODE_VISITOR, TRAVERSE_ALL_CHILDREN) {}

void FlagRemover::apply(osg::Geode &node) {
	if(node.getName() == "sponza_04") {
		node.removeDrawables(0, node.getNumDrawables());
	}
}

LightAnimationCallback::LightAnimationCallback() : lastFrame(0.0), simulationTime(0.0) {}

void LightAnimationCallback::operator() (osg::Node *node, osg::NodeVisitor *nv) {
	osg::MatrixTransform* transform = dynamic_cast<osg::MatrixTransform*>(node);
	double time = nv->getFrameStamp()->getSimulationTime();
	if(config->getUseAnimatedLight()) {
		simulationTime += (time - lastFrame) * config->getLightAnimationSpeed();
	}

	if(transform) {
		osg::Vec3f pos = config->getLightCenter();
		osg::Matrixd origin = osg::Matrixd::translate(pos);
		switch(config->getLightAnimationType()) {
		case SceneConfig::ANIMATE_STATIC:
			transform->setMatrix(origin);
			break;
		case SceneConfig::ANIMATE_ZAXIS:
			transform->setMatrix(origin * osg::Matrixd::translate(0.0f, 0.0f, LIGHT_VARIANCE_Z * cos(simulationTime)));
			break;
		case SceneConfig::ANIMATE_ROTATE:
			transform->setMatrix(origin * osg::Matrixd::rotate(simulationTime, 0.0f, 0.0f, 1.0f));
			break;
		case SceneConfig::ANIMATE_YAXIS:
			transform->setMatrix(origin * osg::Matrixd::translate(0.0f, LIGHT_VARIANCE_Y * cos(simulationTime), 0.0f));
			break;
		}
	}
	lastFrame = time;
}
