#include <stdlib.h>
#include <iostream>
#include "FPSManipulator.h"
#include <osgUtil/LineSegmentIntersector>
#include <osgViewer/GraphicsWindow>
#include <osg/Notify>

using namespace osg;
using namespace osgGA;

FPSManipulator::FPSManipulator() :
		keyW(false), keyA(false), keyS(false), keyD(false), keyShift(false),
		mouseSenseX(1.5), mouseSenseY(1.5),
		moveSenseForward(2000.0), moveSenseSideways(1000.0),
		shiftFactor(0.2) {
}

void FPSManipulator::setByMatrix(const Matrixd& matrix) {
	_eye = matrix.getTrans();
	_rotation = matrix.getRotate();

	if (getVerticalAxisFixed()) {
		fixVerticalAxis(_eye, _rotation, true);
	}
}

void FPSManipulator::setByInverseMatrix(const Matrixd& matrix) {
	setByMatrix(Matrixd::inverse(matrix));
}

Matrixd FPSManipulator::getMatrix() const {
	return Matrixd::rotate(_rotation) * Matrixd::translate(_eye);
}

Matrixd FPSManipulator::getInverseMatrix() const {
	return Matrixd::translate(-_eye) * Matrixd::rotate(_rotation.inverse());
}

void FPSManipulator::setTransformation(const osg::Vec3d& eye,
		const osg::Quat& rotation) {
	_eye = eye;
	_rotation = rotation;

	if (getVerticalAxisFixed()) {
		fixVerticalAxis(_eye, _rotation, true);
	}
}

void FPSManipulator::getTransformation(osg::Vec3d& eye,
		osg::Quat& rotation) const {
	eye = _eye;
	rotation = _rotation;
}

void FPSManipulator::setTransformation(const osg::Vec3d& eye,
		const osg::Vec3d& center, const osg::Vec3d& up) {
	osg::Matrixd m(osg::Matrixd::lookAt(eye, center, up));
	_eye = eye;
	_rotation = m.getRotate().inverse();

	if (getVerticalAxisFixed()) {
		fixVerticalAxis(_eye, _rotation, true);
	}
}

void FPSManipulator::getTransformation(osg::Vec3d& eye, osg::Vec3d& center,
		osg::Vec3d& up) const {
	center = _eye + _rotation * osg::Vec3d(0., 0., -1.);
	eye = _eye;
	up = _rotation * osg::Vec3d(0., 1., 0.);
}

bool FPSManipulator::handle(const GUIEventAdapter& ea, GUIActionAdapter& us) {
	bool ret = StandardManipulator::handle(ea, us);
	us.requestContinuousUpdate(true);
	return ret;
}

bool FPSManipulator::handleFrame(const GUIEventAdapter& ea,
		GUIActionAdapter& us) {
	bool ret = StandardManipulator::handleFrame(ea, us);
	if ((keyW || keyA || keyS || keyD)
			&& performKeyMovement(_delta_frame_time)) {
		us.requestRedraw();
	}
	return ret;
}

bool FPSManipulator::handleMouseMove(const GUIEventAdapter& ea,
		GUIActionAdapter& us) {
	addMouseEvent(ea);

	const osg::GraphicsContext* gc = ea.getGraphicsContext();
	const osgViewer::GraphicsWindow* gw =
			dynamic_cast<const osgViewer::GraphicsWindow*>(gc);

	float x = ea.getWindowWidth() / 2.0f;
	float y = ea.getWindowHeight() / 2.0f;
	float nx = ea.getX();
	float ny = ea.getY();

	if(nx == x && ny == y) {
		flushMouseEventStack();
	} else if (performMovement()) {
		us.requestRedraw();

		// Pin mouse pointer to the center of the screen
		if (osg::Vec2f(x-nx, y-ny).length() > 200.0f && gw) {
			const_cast<osgViewer::GraphicsWindow*>(gw)->requestWarpPointer(x,
					y);
			flushMouseEventStack();
		}
	}

	us.requestContinuousUpdate(false);
	_thrown = false;

	return true;
}

bool FPSManipulator::handleMouseDrag(const GUIEventAdapter& ea,
		GUIActionAdapter& us) {
	addMouseEvent(ea);

	if (performMovement()) {
		us.requestRedraw();

		// Pin mouse pointer to the center of the screen.
		const osg::GraphicsContext* gc = ea.getGraphicsContext();
		const osgViewer::GraphicsWindow* gw =
				dynamic_cast<const osgViewer::GraphicsWindow*>(gc);

		unsigned int x = ea.getWindowWidth() / 2;
		unsigned int y = ea.getWindowHeight() / 2;
		if (gw) {
			const_cast<osgViewer::GraphicsWindow*>(gw)->requestWarpPointer(x,
					y);
			flushMouseEventStack();
		}
	}

	us.requestContinuousUpdate(false);
	_thrown = false;

	return true;
}

bool FPSManipulator::handleKeyDown(const GUIEventAdapter& ea,
		GUIActionAdapter& us) {
	bool ret = StandardManipulator::handleKeyDown(ea, us);
	if (!ret) {
		switch (ea.getKey()) {
		case 'w':
		case 'W':
			keyW = true;
			break;
		case 'a':
		case 'A':
			keyA = true;
			break;
		case 's':
		case 'S':
			keyS = true;
			break;
		case 'd':
		case 'D':
			keyD = true;
			break;
		case GUIEventAdapter::KEY_Shift_L:
			keyShift = true;
			break;
		}
	}
	return ret;
}

bool FPSManipulator::handleKeyUp(const GUIEventAdapter& ea,
		GUIActionAdapter& us) {
	bool ret = StandardManipulator::handleKeyUp(ea, us);
	if (!ret) {
		switch (ea.getKey()) {
		case 'w':
		case 'W':
			keyW = false;
			break;
		case 'a':
		case 'A':
			keyA = false;
			break;
		case 's':
		case 'S':
			keyS = false;
			break;
		case 'd':
		case 'D':
			keyD = false;
			break;
		case GUIEventAdapter::KEY_Shift_L:
			keyShift = false;
			break;
		}
	}
	return ret;
}

void FPSManipulator::flushMouseEventStack() {
	_ga_t1 = NULL;
	_ga_t0 = NULL;
}

bool FPSManipulator::performMovement() {
	if (_ga_t0.get() == NULL || _ga_t1.get() == NULL)
		return false;

	double eventTimeDelta = _ga_t0->getTime() - _ga_t1->getTime();
	if (eventTimeDelta < 0.) {
		OSG_WARN << "Manipulator warning: eventTimeDelta = " << eventTimeDelta
					<< std::endl;
		eventTimeDelta = 0.;
	}

	float dx = _ga_t0->getXnormalized() - _ga_t1->getXnormalized();
	float dy = _ga_t0->getYnormalized() - _ga_t1->getYnormalized();

	if (dx == 0. && dy == 0.)
		return false;

	bool ret = false;
	unsigned int buttonMask = _ga_t1->getButtonMask();
	if (buttonMask == GUIEventAdapter::LEFT_MOUSE_BUTTON) {
		ret = performMovementLeftMouseButton(eventTimeDelta, dx, dy);
	} else if (buttonMask == GUIEventAdapter::MIDDLE_MOUSE_BUTTON
			|| buttonMask
					== (GUIEventAdapter::LEFT_MOUSE_BUTTON
							| GUIEventAdapter::RIGHT_MOUSE_BUTTON)) {
		ret = performMovementMiddleMouseButton(eventTimeDelta, dx, dy);
	} else if (buttonMask == GUIEventAdapter::RIGHT_MOUSE_BUTTON) {
		ret = performMovementRightMouseButton(eventTimeDelta, dx, dy);
	}

	bool md = performMouseDeltaMovement(dx, dy);
	return ret || md;
}

bool FPSManipulator::performKeyMovement(double dt) {
	bool ret = false;
	double mul = keyShift ? shiftFactor : 1.0;
	if (keyW) {
		moveForward(mul*moveSenseForward * dt);
		ret = true;
	}
	if (keyS) {
		moveForward(-mul*moveSenseForward * dt);
		ret = true;
	}
	if (keyA) {
		moveRight(-mul*moveSenseSideways * dt);
		ret = true;
	}
	if (keyD) {
		moveRight(mul*moveSenseSideways * dt);
		ret = true;
	}

	return ret;
}

bool FPSManipulator::performMouseDeltaMovement(const float dx, const float dy) {
	if (getVerticalAxisFixed()) {
		CoordinateFrame coordinateFrame = getCoordinateFrame(_eye);
		Vec3d localUp = getUpVector(coordinateFrame);

		rotateYawPitch(_rotation, mouseSenseX*dx, mouseSenseY*dy, localUp);
	} else {
		rotateYawPitch(_rotation, mouseSenseX*dx, mouseSenseY*dy);
	}
	return true;
}

void FPSManipulator::moveForward(const double distance) {
	moveForward(_rotation, distance);
}

void FPSManipulator::moveForward(const Quat& rotation, const double distance) {
	_eye += rotation * Vec3d(0., 0., -distance);
}

void FPSManipulator::moveRight(const double distance) {
	_eye += _rotation * osg::Vec3d(distance, 0., 0.);
}

void FPSManipulator::setMouseSense(double sx, double sy) {
	mouseSenseX = sx;
	mouseSenseY = sy;
}

void FPSManipulator::setMoveSense(double forward, double sideways) {
	moveSenseForward = forward;
	moveSenseSideways = sideways;
}

void FPSManipulator::setShiftFactor(double factor) {
	shiftFactor = factor;
}
