#include "SceneConfig.h"
#include "Defines.h"

#include <iostream>
#include <sstream>

const float SceneConfig::scaleFactor = 1.1f;

SceneConfig::SceneConfig() {
	stateId = 1;
	dumpId = 0;

	currentSection = SceneConfig::SECTION_GENERAL;

	optionsText = 0;
	valuesText = 0;

	debugSHNode = 0;
	debugNode = 0;

	config.lpvPropagationSteps = 4;
	uniform.lpvPropagationSteps = new osg::Uniform("config_lpvPropagationSteps", config.lpvPropagationSteps);

	config.lpvPropagationFactor = 3.0f;
	uniform.lpvPropagationFactor = new osg::Uniform("config_lpvPropagationFactor", config.lpvPropagationFactor);

	config.lpvInjectionFactor = 20.0f;
	uniform.lpvInjectionFactor = new osg::Uniform("config_lpvInjectionFactor", config.lpvInjectionFactor);

	config.lpvRenderFactor = 20.0f;
	uniform.lpvRenderFactor = new osg::Uniform("config_lpvRenderFactor", config.lpvRenderFactor);

	config.gvOcclusionFactor = 20.0f;
	uniform.gvOcclusionFactor = new osg::Uniform("config_gvOcclusionFactor", config.gvOcclusionFactor);

	config.gvInjectionFactor = 20.0f;
	uniform.gvInjectionFactor = new osg::Uniform("config_gvInjectionFactor", config.gvInjectionFactor);

	config.lpvSampleLevel = -1;

	config.useDirectLightning = true;
	uniform.useDirectLightning = new osg::Uniform("config_useDirectLightning", config.useDirectLightning);

	config.useIndirectLightning = true;
	uniform.useIndirectLightning = new osg::Uniform("config_useIndirectLightning", config.useIndirectLightning);

	config.useDynamicLpvs = true;
	uniform.useDynamicLpvs = new osg::Uniform("config_useDynamicLpvs", config.useDynamicLpvs);

	config.useDiffuseMaps = true;
	uniform.useDiffuseMaps = new osg::Uniform("config_useDiffuseMaps", config.useDiffuseMaps);

	config.useBumpMaps = true;
	uniform.useBumpMaps = new osg::Uniform("config_useBumpMaps", config.useBumpMaps);

	config.useShadowMaps = true;
	uniform.useShadowMaps = new osg::Uniform("config_useShadowMaps", config.useShadowMaps);

	config.useIndirectOcclusion = true;
	uniform.useIndirectOcclusion = new osg::Uniform("config_useIndirectOcclusion", config.useIndirectOcclusion);

	config.useDebugGrid = false;
	uniform.useDebugGrid = new osg::Uniform("config_useDebugGrid", config.useDebugGrid);

	config.useDebugSH = false;
	uniform.useDebugSH = new osg::Uniform("config_useDebugSH", config.useDebugSH);


	config.lightAnimationSpeed = 0.1f;
	config.useAnimatedLight = true;
	config.lightAnimationType = SceneConfig::ANIMATE_YAXIS;
	config.lightCenter = osg::Vec3f(LIGHT_CENTER_X, LIGHT_CENTER_Y, LIGHT_CENTER_Z);

	config.debugMode = DEBUG_PROPAGATION;
	uniform.debugMode = new osg::Uniform("config_debugMode", config.debugMode);
}


void SceneConfig::setLpvPropagationSteps(int lpvPropagationSteps) {
	uniform.lpvPropagationSteps->set(config.lpvPropagationSteps = lpvPropagationSteps);
	onChangedState();
}

int SceneConfig::getLpvPropagationSteps() const {
	return config.lpvPropagationSteps;
}


void SceneConfig::setLpvPropagationFactor(float lpvPropagationFactor) {
	uniform.lpvPropagationFactor->set(config.lpvPropagationFactor = lpvPropagationFactor);
	onChangedState();
}

float SceneConfig::getLpvPropagationFactor() const {
	return config.lpvPropagationFactor;
}


void SceneConfig::setLpvInjectionFactor(float lpvInjectionFactor) {
	uniform.lpvInjectionFactor->set(config.lpvInjectionFactor = lpvInjectionFactor);
	onChangedState();
}

float SceneConfig::getLpvInjectionFactor() const {
	return config.lpvInjectionFactor;
}


void SceneConfig::setLpvRenderFactor(float lpvRenderFactor) {
	uniform.lpvRenderFactor->set(config.lpvRenderFactor = lpvRenderFactor);
	onChangedState();
}

float SceneConfig::getLpvRenderFactor() const {
	return config.lpvRenderFactor;
}


void SceneConfig::setGvOcclusionFactor(float gvOcclusionFactor) {
	uniform.gvOcclusionFactor->set(config.gvOcclusionFactor = gvOcclusionFactor);
	onChangedState();
}

float SceneConfig::getGvOcclusionFactor() const {
	return config.gvOcclusionFactor;
}

void SceneConfig::setGvInjectionFactor(float gvInjectionFactor) {
	uniform.gvInjectionFactor->set(config.gvInjectionFactor = gvInjectionFactor);
	onChangedState();
}

float SceneConfig::getGvInjectionFactor() const {
	return config.gvInjectionFactor;
}

void SceneConfig::setLpvSampleLevel(int level) {
	config.lpvSampleLevel = level;
	onChangedState();
}

int SceneConfig::getLpvSampleLevel() const {
	return config.lpvSampleLevel;
}


void SceneConfig::setUseDirectLightning(bool useDirectLightning) {
	uniform.useDirectLightning->set(config.useDirectLightning = useDirectLightning);
	onChangedState();
}

bool SceneConfig::getUseDirectLightning() const {
	return config.useDirectLightning;
}


void SceneConfig::setUseIndirectLightning(bool useIndirectLightning) {
	uniform.useIndirectLightning->set(config.useIndirectLightning = useIndirectLightning);
	onChangedState();
}

bool SceneConfig::getUseIndirectLightning() const {
	return config.useIndirectLightning;
}

void SceneConfig::setUseDynamicLpvs(bool useDynamicLpvs) {
	uniform.useDynamicLpvs->set(config.useDynamicLpvs = useDynamicLpvs);
	onChangedState();
}
bool SceneConfig::getUseDynamicLpvs() const {
	return config.useDynamicLpvs;
}


void SceneConfig::setUseDiffuseMaps(bool useDiffuseMaps) {
	uniform.useDiffuseMaps->set(config.useDiffuseMaps = useDiffuseMaps);
	onChangedState();
}

bool SceneConfig::getUseDiffuseMaps() const {
	return config.useDiffuseMaps;
}


void SceneConfig::setUseBumpMaps(bool useBumpMaps) {
	uniform.useBumpMaps->set(config.useBumpMaps = useBumpMaps);
	onChangedState();
}

bool SceneConfig::getUseBumpMaps() const {
	return config.useBumpMaps;
}


void SceneConfig::setUseShadowMaps(bool useShadowMaps) {
	uniform.useShadowMaps->set(config.useShadowMaps = useShadowMaps);
	onChangedState();
}

bool SceneConfig::getUseShadowMaps() const {
	return config.useShadowMaps;
}


void SceneConfig::setUseIndirectOcclusion(bool useIndirectOcclusion) {
	uniform.useIndirectOcclusion->set(config.useIndirectOcclusion = useIndirectOcclusion);
	onChangedState();
}

bool SceneConfig::getUseIndirectOcclusion() const {
	return config.useIndirectOcclusion;
}


void SceneConfig::setUseDebugGrid(bool useDebugGrid) {
	uniform.useDebugGrid->set(config.useDebugGrid = useDebugGrid);
	onChangedState();
}

bool SceneConfig::getUseDebugGrid() const {
	return config.useDebugGrid;
}

void SceneConfig::setUseDebugSH(bool useDebugSH) {
	uniform.useDebugSH->set(config.useDebugSH = useDebugSH);
	onChangedState();
}

bool SceneConfig::getUseDebugSH() const {
	return config.useDebugSH;
}

void SceneConfig::setLightAnimationSpeed(float lightAnimationSpeed) {
	config.lightAnimationSpeed = lightAnimationSpeed;
	onChangedState();
}

float SceneConfig::getLightAnimationSpeed() const {
	return config.lightAnimationSpeed;
}

void SceneConfig::setUseAnimatedLight(bool useAnimatedLight) {
	config.useAnimatedLight = useAnimatedLight;
	onChangedState();
}

bool SceneConfig::getUseAnimatedLight() const {
	return config.useAnimatedLight;
}

void SceneConfig::setLightAnimationType(LightAnimationType lightAnimationType) {
	config.lightAnimationType = lightAnimationType;
	onChangedState();
}

SceneConfig::LightAnimationType SceneConfig::getLightAnimationType() const {
	return config.lightAnimationType;
}

void SceneConfig::setLightCenter(const osg::Vec3& lightCenter) {
	config.lightCenter = lightCenter;
	onChangedState();
}
osg::Vec3 SceneConfig::getLightCenter() const {
	return config.lightCenter;
}


void SceneConfig::setDebugMode(SceneConfig::DebugMode debugMode) {
	uniform.debugMode->set(config.debugMode = debugMode);
	onChangedState();
}
SceneConfig::DebugMode SceneConfig::getDebugMode() const {
	return config.debugMode;
}


void SceneConfig::changeSection() {
	int sid = currentSection;
	sid = (sid+1) % SECTION_LAST;
	currentSection = static_cast<SceneConfig::HudSection>(sid);
	onChangedState();
}

void SceneConfig::install(osg::StateSet* ss) {
	ss->addUniform(uniform.lpvPropagationSteps);
	ss->addUniform(uniform.lpvPropagationFactor);
	ss->addUniform(uniform.lpvInjectionFactor);
	ss->addUniform(uniform.lpvRenderFactor);
	ss->addUniform(uniform.gvOcclusionFactor);
	ss->addUniform(uniform.gvInjectionFactor);
	ss->addUniform(uniform.debugMode);

	ss->addUniform(uniform.useDirectLightning);
	ss->addUniform(uniform.useIndirectLightning);
	ss->addUniform(uniform.useDynamicLpvs);
	ss->addUniform(uniform.useDiffuseMaps);
	ss->addUniform(uniform.useBumpMaps);
	ss->addUniform(uniform.useShadowMaps);
	ss->addUniform(uniform.useIndirectOcclusion);
	ss->addUniform(uniform.useDebugGrid);
	ss->addUniform(uniform.useDebugSH);
}

int SceneConfig::getStateId() const {
	return stateId;
}

int SceneConfig::getDumpId() const {
	return dumpId;
}

void SceneConfig::requestDump() {
	dumpId++;
	onChangedState();
	std::cout << "Requested a dump..." << std::endl;
}

bool SceneConfig::handle(const osgGA::GUIEventAdapter& ea, osgGA::GUIActionAdapter& aa, osg::Object* obj, osg::NodeVisitor* nv) {
	osgGA::GUIEventAdapter::EventType ev = ea.getEventType();

	if(
		ev != osgGA::GUIEventAdapter::KEYDOWN
	) return false;

	int key     = ea.getKey();
	int keyMask = ea.getModKeyMask();

	if(key == -1) return false;

	if(key == osgGA::GUIEventAdapter::KEY_Tab) {
		changeSection();
	} else if(key == osgGA::GUIEventAdapter::KEY_Insert) {
		requestDump();
	} else if(currentSection == SceneConfig::SECTION_GENERAL) {
		switch(key) {

		/*
		 * Enable and disable various features using the number keys.
		 */
		case osgGA::GUIEventAdapter::KEY_F1:
			setUseDirectLightning(!getUseDirectLightning());
			return true;
		case osgGA::GUIEventAdapter::KEY_F2:
			setUseIndirectLightning(!getUseIndirectLightning());
			return true;
		case osgGA::GUIEventAdapter::KEY_F3:
			setUseDiffuseMaps(!getUseDiffuseMaps());
			return true;
		case osgGA::GUIEventAdapter::KEY_F4:
			setUseBumpMaps(!getUseBumpMaps());
			return true;
		case osgGA::GUIEventAdapter::KEY_F5:
			setUseShadowMaps(!getUseShadowMaps());
			return true;
		case osgGA::GUIEventAdapter::KEY_F6:
			setUseIndirectOcclusion(!getUseIndirectOcclusion());
			return true;
		case osgGA::GUIEventAdapter::KEY_F7:
			setUseDynamicLpvs(!getUseDynamicLpvs());
			return true;
		case osgGA::GUIEventAdapter::KEY_F8:
			setUseDebugGrid(!getUseDebugGrid());
			return true;
		case osgGA::GUIEventAdapter::KEY_F9:
			setUseDebugSH(!getUseDebugSH());
			return true;
		case osgGA::GUIEventAdapter::KEY_F10:
			setDebugMode(static_cast<SceneConfig::DebugMode>((getDebugMode() + 1) % DEBUG_LAST));
			return true;

		/*
		 * Change the amount of iterations used during propagation using the plus and minus keys.
		 */
		case osgGA::GUIEventAdapter::KEY_KP_Add:
			setLpvPropagationSteps(getLpvPropagationSteps() + 1);
			return true;
		case osgGA::GUIEventAdapter::KEY_KP_Subtract:
			setLpvPropagationSteps(std::max(1, getLpvPropagationSteps() - 1));
			return true;

		/*
		 * Double or halve various factors by using the numpad keys. Uppermost row doubles
		 * a value while the middle row halves the value. Each column corresponds to a
		 * specific factor in the order:
		 * 7/4 - Injection factor
		 * 8/5 - Propagation factor
		 * 9/6 - Render factor
		 */
		case osgGA::GUIEventAdapter::KEY_7:
			setLpvInjectionFactor(scaleFactor * getLpvInjectionFactor());
			return true;
		case osgGA::GUIEventAdapter::KEY_4:
			setLpvInjectionFactor(1.0f/scaleFactor * getLpvInjectionFactor());
			return true;

		case osgGA::GUIEventAdapter::KEY_8:
			setLpvPropagationFactor(scaleFactor * getLpvPropagationFactor());
			return true;
		case osgGA::GUIEventAdapter::KEY_5:
			setLpvPropagationFactor(1.0f/scaleFactor * getLpvPropagationFactor());
			return true;

		case osgGA::GUIEventAdapter::KEY_9:
			setLpvRenderFactor(scaleFactor * getLpvRenderFactor());
			return true;
		case osgGA::GUIEventAdapter::KEY_6:
			setLpvRenderFactor(1.0f/scaleFactor * getLpvRenderFactor());
			return true;

		case osgGA::GUIEventAdapter::KEY_3:
			setGvOcclusionFactor(scaleFactor * getGvOcclusionFactor());
			return true;
		case osgGA::GUIEventAdapter::KEY_Comma:
			setGvOcclusionFactor(1.0f/scaleFactor * getGvOcclusionFactor());
			return true;

		case osgGA::GUIEventAdapter::KEY_2:
			setGvInjectionFactor(scaleFactor * getGvInjectionFactor());
			return true;
		case osgGA::GUIEventAdapter::KEY_1:
			setGvInjectionFactor(1.0f/scaleFactor * getGvInjectionFactor());
			return true;

		/*
		 * Change which level of the octree to sample from when rendering.
		 */
		case osgGA::GUIEventAdapter::KEY_Page_Up:
			setLpvSampleLevel(getLpvSampleLevel() + 1);
			return true;
		case osgGA::GUIEventAdapter::KEY_Page_Down:
			setLpvSampleLevel(std::max(-1, getLpvSampleLevel() - 1));
			return true;

		default:
			return false;
		}
	} else if(currentSection == SceneConfig::SECTION_LIGHTS) {
		switch(key) {
		/*
		 * Light movement
		 */
		case osgGA::GUIEventAdapter::KEY_8:
			setLightCenter(getLightCenter() + osg::Vec3f(0.0f, LIGHT_MOVEMENT_SPEED, 0.0f));
			return true;
		case osgGA::GUIEventAdapter::KEY_2:
			setLightCenter(getLightCenter() + osg::Vec3f(0.0f, -LIGHT_MOVEMENT_SPEED, 0.0f));
			return true;
		case osgGA::GUIEventAdapter::KEY_6:
			setLightCenter(getLightCenter() + osg::Vec3f(LIGHT_MOVEMENT_SPEED, 0.0f, 0.0f));
			return true;
		case osgGA::GUIEventAdapter::KEY_4:
			setLightCenter(getLightCenter() + osg::Vec3f(-LIGHT_MOVEMENT_SPEED, 0.0f, 0.0f));
			return true;
		case osgGA::GUIEventAdapter::KEY_9:
			setLightCenter(getLightCenter() + osg::Vec3f(0.0f, 0.0f, LIGHT_MOVEMENT_SPEED));
			return true;
		case osgGA::GUIEventAdapter::KEY_3:
			setLightCenter(getLightCenter() + osg::Vec3f(0.0f, 0.0f, -LIGHT_MOVEMENT_SPEED));
			return true;
		case osgGA::GUIEventAdapter::KEY_5:
			setLightCenter(osg::Vec3f(LIGHT_CENTER_X, LIGHT_CENTER_Y, LIGHT_CENTER_Z));
			return true;


		/*
		 * Animation control
		 */
		case osgGA::GUIEventAdapter::KEY_KP_Add:
			setLightAnimationSpeed(scaleFactor * getLightAnimationSpeed());
			return true;
		case osgGA::GUIEventAdapter::KEY_KP_Subtract:
			setLightAnimationSpeed(1.0f/scaleFactor * getLightAnimationSpeed());
			return true;
		case osgGA::GUIEventAdapter::KEY_F1:
			setLightAnimationType(SceneConfig::ANIMATE_STATIC);
			return true;
		case osgGA::GUIEventAdapter::KEY_F2:
			setLightAnimationType(SceneConfig::ANIMATE_ZAXIS);
			return true;
		case osgGA::GUIEventAdapter::KEY_F3:
			setLightAnimationType(SceneConfig::ANIMATE_ROTATE);
			return true;
		case osgGA::GUIEventAdapter::KEY_F4:
			setLightAnimationType(SceneConfig::ANIMATE_YAXIS);
			return true;
		case osgGA::GUIEventAdapter::KEY_F12:
			setUseAnimatedLight(!getUseAnimatedLight());
			return true;

		default:
			return false;
		}
	}

}

void SceneConfig::setHudOptionsText(osgText::Text* optionsText) {
	this->optionsText = optionsText;
	updateHudOptionsText();
}

void SceneConfig::setHudValuesText(osgText::Text* valuesText) {
	this->valuesText = valuesText;
	updateHudValuesText();
}

void SceneConfig::setDebugNode(osg::Node* debugNode) {
	this->debugNode = debugNode;
	onChangedState();
}

void SceneConfig::setDebugSHNode(osg::Node* debugSHNode) {
	this->debugSHNode = debugSHNode;
	onChangedState();
}

void SceneConfig::onChangedState() {
	stateId++;
	updateHudOptionsText();
	updateHudValuesText();
	if(debugNode) {
		debugNode->setNodeMask(config.useDebugGrid ? ~0 : 0);
	}
	if(debugSHNode) {
		debugSHNode->setNodeMask(config.useDebugSH ? ~0 : 0);
	}
}

void SceneConfig::updateHudOptionsText() {
	if(optionsText) {
		std::stringstream ss;
		ss << "Insert - Dump LPVs to file" << std::endl;
		if(currentSection == SceneConfig::SECTION_GENERAL) {
			ss << "F1 - Direct Lighting: " << std::endl;
			ss << "F2 - Indirect Lighting: " << std::endl;
			ss << "F3 - Diffuse Maps: " << std::endl;
			ss << "F4 - Bump Maps: " << std::endl;
			ss << "F5 - Shadow Maps: " << std::endl;
			ss << "F6 - Indirect Occlusion: " << std::endl;
			ss << "F7 - Dynamic LPVs: " << std::endl;
			ss << "F8 - Debug Grid: " << std::endl;
			ss << "F9 - Debug SH: " << std::endl;
			ss << "F10 - Debug Mode: " << std::endl;
			ss << "7/4 - Injection factor: " << std::endl;
			ss << "8/5 - Propagation factor: " << std::endl;
			ss << "9/6 - Render factor: " << std::endl;
			ss << "3/, - Occlusion factor: " << std::endl;
			ss << "2/1 - Geometry factor: " << std::endl;
			ss << "PgUp/Dn - Sample level: " << std::endl;
			ss << "+/- - Propagation steps: " << std::endl;
		} else if(currentSection == SceneConfig::SECTION_LIGHTS) {
			ss << "4/6 - Light center x: " << std::endl;
			ss << "8/2 - Light center y: " << std::endl;
			ss << "9/3 - Light center z: " << std::endl;
			ss << "5 - Reset light center " << std::endl;
			ss << std::endl;
			ss << "+/- - Light speed: " << std::endl;
			ss << "Light animation type: " << std::endl;
			ss << "F12 - Animate lights: " << std::endl;
			ss << "F1 - Static lights" << std::endl;
			ss << "F2 - Z-Axis animation" << std::endl;
			ss << "F3 - Rotating animation" << std::endl;
			ss << "F4 - Y-Axis animation" << std::endl;
		}
		optionsText->setText(ss.str());
	}
}

void SceneConfig::updateHudValuesText() {
	if(valuesText) {
		std::stringstream ss;
		ss << std::endl;
		if(currentSection == SceneConfig::SECTION_GENERAL) {
			ss << (config.useDirectLightning 	? "on" : "off") << std::endl;
			ss << (config.useIndirectLightning 	? "on" : "off") << std::endl;
			ss << (config.useDiffuseMaps 		? "on" : "off") << std::endl;
			ss << (config.useBumpMaps 			? "on" : "off") << std::endl;
			ss << (config.useShadowMaps 		? "on" : "off") << std::endl;
			ss << (config.useIndirectOcclusion 	? "on" : "off") << std::endl;
			ss << (config.useDynamicLpvs 		? "on" : "off") << std::endl;
			ss << (config.useDebugGrid	 		? "on" : "off") << std::endl;
			ss << (config.useDebugSH			? "on" : "off") << std::endl;

			switch(config.debugMode) {
			case DEBUG_GEOMETRY: 	ss << "Geometry" 	<< std::endl; break;
			case DEBUG_INJECTION: 	ss << "Injection" 	<< std::endl; break;
			case DEBUG_PROPAGATION:	ss << "Propagation"	<< std::endl; break;
			}

			ss << config.lpvInjectionFactor 	<< std::endl;
			ss << config.lpvPropagationFactor 	<< std::endl;
			ss << config.lpvRenderFactor 		<< std::endl;
			ss << config.gvOcclusionFactor		<< std::endl;
			ss << config.gvInjectionFactor		<< std::endl;
			if(config.lpvSampleLevel == -1) {
				ss << "Auto"					<< std::endl;
			} else {
				ss << config.lpvSampleLevel		<< std::endl;
			}
			ss << config.lpvPropagationSteps	<< std::endl;
		} else if(currentSection == SceneConfig::SECTION_LIGHTS) {
			ss << config.lightCenter.x() << std::endl;
			ss << config.lightCenter.y() << std::endl;
			ss << config.lightCenter.z() << std::endl;
			ss << std::endl;
			ss << std::endl;
			ss << config.lightAnimationSpeed	<< std::endl;
			ss << (config.useAnimatedLight		? "on" : "off") << std::endl;
			switch(config.lightAnimationType) {
			case SceneConfig::ANIMATE_STATIC: ss << "Static" << std::endl; break;
			case SceneConfig::ANIMATE_ZAXIS: ss << "Z-Axis" << std::endl; break;
			case SceneConfig::ANIMATE_ROTATE: ss << "Rotate" << std::endl; break;
			case SceneConfig::ANIMATE_YAXIS: ss << "Y-Axis" << std::endl; break;
			}
		}
		valuesText->setText(ss.str());
	}
}



SceneConfigurable::SceneConfigurable() : config(0), lastStateId(0), lastDumpId(0) {}

void SceneConfigurable::setConfig(SceneConfig* config) {
	this->config = config;
}

SceneConfig* SceneConfigurable::getConfig() const {
	return config;
}

bool SceneConfigurable::hasChangedState() const {
	return config->getStateId() > lastStateId;
}

void SceneConfigurable::updateState() {
	lastStateId = config->getStateId();
}

bool SceneConfigurable::hasRequestedDump() const {
	return config->getDumpId() > lastDumpId;
}

void SceneConfigurable::updateDump() {
	lastDumpId = config->getDumpId();
}

std::string SceneConfigurable::getDumpId() const {
	std::stringstream ss;
	ss << "dump" << config->getDumpId();
	return ss.str();
}
