#include "Volume.h"

#include <osg/Texture>
#include <sstream>
#include <fstream>
#include <iostream>

namespace data {

Volume::Volume(int size, const std::string& name) : size(size), name(name) {
	texture = new osgCuda::Texture3D;
	texture->addIdentifier(name);
	texture->setName(name);
	texture->setUsage(osgCompute::GL_SOURCE_COMPUTE_TARGET);
	texture->setTextureSize(size, size, size);
	texture->setInternalFormat(GL_RGBA32F_ARB);
	texture->setSourceFormat(GL_RGBA);
	texture->setSourceType(GL_FLOAT);
	texture->setFilter(osg::Texture::MIN_FILTER, osg::Texture::LINEAR);
	texture->setFilter(osg::Texture::MAG_FILTER, osg::Texture::LINEAR);
	/*texture->setFilter(osg::Texture::MIN_FILTER, osg::Texture::NEAREST);
	texture->setFilter(osg::Texture::MAG_FILTER, osg::Texture::NEAREST);*/
	memory = texture->getMemory();

	// Initialize the texture memory here in order to provide useful error messages upon failure.
	if(!memory->init()) {
		std::cerr << "Volume: Failed to initialize volume memory for " << name << "." << std::endl;
	}
}

Volume::~Volume() {}

int Volume::getSize() const {
	return size;
}

osgCuda::Texture3D* Volume::getTexture() const {
	return texture;
}

osgCompute::Memory* Volume::getMemory() const {
	return memory;
}

void Volume::dump(const std::string& path, const std::string& dumpId, const std::string& postfix) const {
	float* data = (float*)memory->map(osgCompute::MAP_HOST_SOURCE);

	std::stringstream ss;
	ss << path << "/" << dumpId << "-" << name;
	if(!postfix.empty()) {
		ss << "-" << postfix;
	}
	ss << ".lpv";
	std::ofstream out(ss.str().c_str());
	out << size << "," << size << "," << size << std::endl;

	std::cout << "Dumping " << name << " to " << ss.str() << std::endl;

	for(int z = 0; z < size; z++) {
		for(int y = 0; y < size; y++) {
			for(int x = 0; x < size; x++) {
				float* sh = data + (z*size*size + y*size + x)*4;
				out << sh[0] << "," << sh[1] << "," << sh[2] << "," << sh[3];
				if(x != size-1) {
					out << ",";
				}
			}
			out << std::endl;
		}
	}
	out.close();
}

} /* namespace data */
