#include <osg/TextureCubeMap>
#include <osg/Texture2D>
#include <osg/MatrixTransform>
#include <osg/PrimitiveSet>
#include <osg/ShapeDrawable>
#include <osg/LineWidth>

#include <osgDB/ReadFile>

#include <osgUtil/TransformCallback>
#include <osgUtil/SmoothingVisitor>

#include <osgText/Text>

#include <osgViewer/Viewer>

#include <iostream>

#include "CalculateBoundingBoxVisitor.h"
#include "Defines.h"
#include "SceneConfig.h"
#include "TangentSpaceVisitor.h"
#include "FPSManipulator.h"
#include "CallbackUtils.h"
#include "LightPrerenderSetup.h"
#include "FrameTimeMonitor.h"
#include "TimeMonitor.h"

int main(int argc, char **argv) {

	data::LightPropagationVolume* lpv = 0; // Temp. For debugging.

	SceneConfig* config = new SceneConfig;

	osgViewer::Viewer viewer;

	osgGA::FPSManipulator* cameraManipulator = new osgGA::FPSManipulator;
	viewer.setCameraManipulator(cameraManipulator);

	viewer.addEventHandler(config);

	FrameTimeMonitor* fps = new FrameTimeMonitor;
	viewer.addEventHandler(fps);


	/*
	 * Ensure that the ambient texture ends up on position 0, diffuse ones on
	 * position 1 and bump maps on position 2 no matter in which order they
	 * are listed or parsed from the .mtl file.
	 */
	std::stringstream optionsStream;
	optionsStream << "AMBIENT=0" << std::endl;
	optionsStream << "DIFFUSE=1" << std::endl;
	optionsStream << "BUMP=2" << std::endl;
	osg::ref_ptr<osgDB::ReaderWriter::Options> options =
			new osgDB::ReaderWriter::Options(optionsStream.str());


	osg::ref_ptr<osg::Node> loadedModel = osgDB::readNodeFile(
			"../Sponza/sponza.obj",
			//"../CubeRoom/CubeRoom.obj",
			options);

	if (loadedModel) {
		std::cout << "Successfully loaded model from file." << std::endl;
	} else {
		std::cerr << "Failed to load scene." << std::endl;
		return 1;
	}

	/*
	 * Remove the huge flag that is blocking the line of sight throughout
	 * the scene.
	 */
	FlagRemover flagRemover;
	loadedModel->accept(flagRemover);

	/*
	 * Ensure that tangents and binormals are generated for the loaded scene.
	 * This is used for bump mapping in the shader.
	 */
	TangentSpaceVisitor tsv;
	loadedModel->traverse(tsv);


	osg::ref_ptr<osg::Group> scene = new osg::Group;

	scene->addChild(loadedModel);

	//constructDebugScene(scene);

	std::cout << "Creating and positioning light sources... " << std::flush;

	//osg::ref_ptr<osg::Group> lights = new osg::Group;
	osg::ref_ptr<osg::MatrixTransform> lights = new osg::MatrixTransform;
	lights->setMatrix(osg::Matrixd::translate(LIGHT_CENTER_X, LIGHT_CENTER_Y, LIGHT_CENTER_Z));

	LightAnimationCallback* lightAnimationCallback = new LightAnimationCallback;
	lightAnimationCallback->setConfig(config);
	lights->addUpdateCallback(lightAnimationCallback);
	//lights->setMatrix(osg::Matrixd::identity());

	/*
	 * This will contain the visual representation of the light sources for debugging.
	 */
	osg::ref_ptr<osg::Geode> lightSpheres = new osg::Geode();
	lights->addChild(lightSpheres);

	for(int i = -1; i <= 1; i++){
		osg::Vec3 pos(i*LIGHT_DISTANCE_X, i*LIGHT_DISTANCE_Y, i*LIGHT_DISTANCE_Z);

		osg::ref_ptr<osg::Light> l = new osg::Light;
		l->setLightNum(0);
		l->setPosition(osg::Vec4(pos.x(), pos.y(), pos.z(), 1.0f));
		l->setAmbient(LIGHT_COLOR);
		l->setDiffuse(LIGHT_COLOR);
		l->setConstantAttenuation(1.0f);

		osg::ref_ptr<osg::LightSource> ls = new osg::LightSource;
		ls->setLight(l);
		ls->setLocalStateSetModes(osg::StateAttribute::ON);

		lights->addChild(ls);


		lightSpheres->addDrawable(new osg::ShapeDrawable(new osg::Sphere(pos, LIGHT_SPHERE_SIZE)));
	}

	std::cout << "[Done]" << std::endl;


	osg::Group* rootNode = new osg::Group();
	config->install(rootNode->getOrCreateStateSet());

	/*
	 * This is the path through the scene that will be taken for the actual render to screen
	 * pass.
	 */

	std::cout << "Setting up scene shaders and main render path." << std::endl;

	osg::Group* realScene = new osg::Group();
	{
#if defined(SCENE_SPONZA)
		realScene->addChild(scene);
		std::string shaderName = "ShadowMapping";
#if defined(DEBUG_LIGHTS)
		{
			osg::StateSet* ss = lightSpheres->getOrCreateStateSet();
			osg::ref_ptr<osg::Program> shaderProgram = new osg::Program;

			osg::ref_ptr<osg::Shader> vs = new osg::Shader(osg::Shader::VERTEX);
			osg::ref_ptr<osg::Shader> fs = new osg::Shader(osg::Shader::FRAGMENT);

			vs->loadShaderSourceFromFile("media/red.vs");
			fs->loadShaderSourceFromFile("media/red.fs");

			shaderProgram->addShader(vs);
			shaderProgram->addShader(fs);

			ss->setAttributeAndModes(shaderProgram, osg::StateAttribute::ON);
		}
#endif
#elif defined(SCENE_DEBUG)
		osg::Geode* geode = new osg::Geode();
		geode->addDrawable(
				new osg::ShapeDrawable(
						new osg::Box(osg::Vec3(0.0f, 0.0f, 0.0f), 1000)));

		realScene->addChild(geode);
		std::string shaderName = "cubemap";
#endif
		osg::StateSet* ss = realScene->getOrCreateStateSet();
		realScene->addChild(lights);
		//realScene->addChild(lightSpheres);
		{
			osg::ref_ptr<osg::Program> shaderProgram = new osg::Program;

			osg::ref_ptr<osg::Shader> vs = new osg::Shader(osg::Shader::VERTEX);
			osg::ref_ptr<osg::Shader> fs = new osg::Shader(osg::Shader::FRAGMENT);

			vs->loadShaderSourceFromFile("media/" + shaderName + ".vs");
			fs->loadShaderSourceFromFile("media/" + shaderName + ".fs");

			shaderProgram->addShader(vs);
			shaderProgram->addShader(fs);

			/*
			 * Map the previously generated tangents and binormals to appropriate vertex attributes.
			 */
			shaderProgram->addBindAttribLocation("osg_Tangent", 6);
			shaderProgram->addBindAttribLocation("osg_Binormal", 7);

			/*
			 * Map the textures to the appropriate shader sampler2D uniforms according to the order defined
			 * by the osgDB::ReaderWriter::Options in the beginning.
			 */
			ss->addUniform(new osg::Uniform("texture_Ambient", 0));
			ss->addUniform(new osg::Uniform("texture_Diffuse", 1));
			ss->addUniform(new osg::Uniform("texture_Normal", 2));

			ss->setAttributeAndModes(shaderProgram, osg::StateAttribute::ON);


			/*
			 * Prepare the scene for deferred rendering.
			 */
			std::cout << "Dispatching light pre-render setup visitor... " << std::flush;
			LightPrerenderSetup lps(config);
			realScene->accept(lps);
			std::cout << "[Done]" << std::endl;

			/*
			 * Finalize and inject the deferred rendering stage into the scene graph.
			 * The relevant uniforms required for subsequent rendering will be registered
			 * to the ss state set.
			 */
			std::cout << "Performing light pre-render setup." << std::endl;
			lps.setup(scene, rootNode, ss);
			lpv = lps.getLPV();
		}
	}

	rootNode->addChild(realScene);


	// Begin HUD.
	{

		osg::Projection* pMatrix = new osg::Projection;
		pMatrix->setMatrix(osg::Matrix::ortho2D(0, VIRTUAL_SCREEN_WIDTH, 0, VIRTUAL_SCREEN_HEIGHT));

		osg::MatrixTransform* mvMatrix = new osg::MatrixTransform;
		mvMatrix->setMatrix(osg::Matrix::translate(osg::Vec3(VIRTUAL_SCREEN_WIDTH - HUD_WIDTH, VIRTUAL_SCREEN_HEIGHT - HUD_HEIGHT, 0.0f)));
		mvMatrix->setReferenceFrame(osg::Transform::ABSOLUTE_RF);

		rootNode->addChild(pMatrix);
		pMatrix->addChild(mvMatrix);

		osg::Geode* hudBackgroundGeode = new osg::Geode();
		{
			osg::Geometry* hudGeometry = new osg::Geometry();

			osg::Vec3Array* vertexArray = new osg::Vec3Array;
			vertexArray->push_back(osg::Vec3(0.0f, 		0.0f, 		-1.0f));
			vertexArray->push_back(osg::Vec3(HUD_WIDTH, 0.0f, 		-1.0f));
			vertexArray->push_back(osg::Vec3(HUD_WIDTH, HUD_HEIGHT, -1.0f));
			vertexArray->push_back(osg::Vec3(0.0f, 		HUD_HEIGHT, -1.0f));

			osg::DrawElementsUInt* indexArray =
			new osg::DrawElementsUInt(osg::PrimitiveSet::POLYGON, 0);
			indexArray->push_back(0);
			indexArray->push_back(1);
			indexArray->push_back(2);
			indexArray->push_back(3);

			osg::Vec2Array* texcoords = new osg::Vec2Array(4);
			(*texcoords)[0].set(0.0f,0.0f);
			(*texcoords)[1].set(1.0f,0.0f);
			(*texcoords)[2].set(1.0f,1.0f);
			(*texcoords)[3].set(0.0f,1.0f);

			osg::Vec4Array* colorArray = new osg::Vec4Array;
			colorArray->push_back(osg::Vec4(0.3f,0.3f,0.3f,0.75f));

			hudGeometry->setTexCoordArray(0,texcoords);
			hudGeometry->addPrimitiveSet(indexArray);
			hudGeometry->setVertexArray(vertexArray);
			hudGeometry->setColorArray(colorArray);
			hudGeometry->setColorBinding(osg::Geometry::BIND_OVERALL);

			hudBackgroundGeode->addDrawable(hudGeometry);

			mvMatrix->addChild(hudBackgroundGeode);


			{
				osg::StateSet* ss = hudBackgroundGeode->getOrCreateStateSet();
				osg::ref_ptr<osg::Program> shaderProgram = new osg::Program;

				osg::ref_ptr<osg::Shader> vs = new osg::Shader(osg::Shader::VERTEX);
				osg::ref_ptr<osg::Shader> fs = new osg::Shader(osg::Shader::FRAGMENT);

				vs->loadShaderSourceFromFile("media/hudBackground.vs");
				fs->loadShaderSourceFromFile("media/hudBackground.fs");

				shaderProgram->addShader(vs);
				shaderProgram->addShader(fs);

				ss->setAttributeAndModes(shaderProgram, osg::StateAttribute::ON);
			}

		}

		osg::Geode* hudTextGeode = new osg::Geode();
		// HUD Text
		{

			float textY = HUD_HEIGHT;

			{ // Information section
				osgText::Text* infoHeader = new osgText::Text;
				infoHeader->setAxisAlignment(osgText::Text::SCREEN);
				infoHeader->setFont("media/liberation-fonts/LiberationSansNarrow-Bold.ttf");
				infoHeader->setCharacterSize(32);
				infoHeader->setPosition(osg::Vec3(20.0f, textY -= infoHeader->getFontHeight(), -0.5f));
				infoHeader->setColor(osg::Vec4(0.0f, 1.0f, 0.0f, 1.0f));
				infoHeader->setText("Information:");
				hudTextGeode->addDrawable(infoHeader);


				osgText::Text* fpsText = new osgText::Text;
				fpsText->setAxisAlignment(osgText::Text::SCREEN);
				fpsText->setFont("media/liberation-fonts/LiberationSansNarrow-Bold.ttf");
				fpsText->setCharacterSize(20);
				fpsText->setPosition(osg::Vec3(20.0f, textY -= fpsText->getFontHeight(), -0.5f));
				fpsText->setColor(osg::Vec4(0.0f, 1.0f, 0.0f, 1.0f));
				fpsText->setText("Frame rate:");
				hudTextGeode->addDrawable(fpsText);


				osgText::Text* fpsValue = new osgText::Text;
				fpsValue->setAxisAlignment(osgText::Text::SCREEN);
				fpsValue->setFont("media/liberation-fonts/LiberationSansNarrow-Bold.ttf");
				fpsValue->setCharacterSize(20);
				fpsValue->setPosition(osg::Vec3(HUD_WIDTH - 75.0f, textY, -0.5f));
				fpsValue->setColor(osg::Vec4(1.0f, 0.0f, 0.0f, 1.0f));
				fps->setFpsValueText(fpsValue);
				hudTextGeode->addDrawable(fpsValue);
			}


			{ // Settings section
				osgText::Text* settingsHeader = new osgText::Text;
				settingsHeader->setAxisAlignment(osgText::Text::SCREEN);
				settingsHeader->setFont("media/liberation-fonts/LiberationSansNarrow-Bold.ttf");
				settingsHeader->setCharacterSize(32);
				settingsHeader->setPosition(osg::Vec3(20.0f, textY -= settingsHeader->getFontHeight(), -0.5f));
				settingsHeader->setColor(osg::Vec4(0.0f, 1.0f, 0.0f, 1.0f));
				settingsHeader->setText("Settings:");
				hudTextGeode->addDrawable(settingsHeader);

				osgText::Text* settingsHelp = new osgText::Text;
				settingsHelp->setAxisAlignment(osgText::Text::SCREEN);
				settingsHelp->setFont("media/liberation-fonts/LiberationSansNarrow-Bold.ttf");
				settingsHelp->setCharacterSize(20);
				settingsHelp->setPosition(osg::Vec3(20.0f, textY -= settingsHelp->getFontHeight(), -0.5f));
				settingsHelp->setColor(osg::Vec4(0.0f, 1.0f, 0.0f, 1.0f));
				settingsHelp->setText("Tab - More settings");
				hudTextGeode->addDrawable(settingsHelp);

				osgText::Text* textOptions = new osgText::Text;
				textOptions->setAxisAlignment(osgText::Text::SCREEN);
				textOptions->setFont("media/liberation-fonts/LiberationSansNarrow-Bold.ttf");
				textOptions->setCharacterSize(20);
				textOptions->setPosition(osg::Vec3(20.0f, textY -= textOptions->getFontHeight(), -0.5f));
				textOptions->setColor(osg::Vec4(0.0f, 1.0f, 0.0f, 1.0f));
				config->setHudOptionsText(textOptions);
				hudTextGeode->addDrawable(textOptions);


				osgText::Text* textValues = new osgText::Text;
				textValues->setAxisAlignment(osgText::Text::SCREEN);
				textValues->setFont("media/liberation-fonts/LiberationSansNarrow-Bold.ttf");
				textValues->setCharacterSize(20);
				textValues->setPosition(osg::Vec3(HUD_WIDTH - 75.0f, textY, -0.5f));
				textValues->setColor(osg::Vec4(1.0f, 0.0f, 0.0f, 1.0f));
				config->setHudValuesText(textValues);
				hudTextGeode->addDrawable(textValues);
			}

			mvMatrix->addChild(hudTextGeode);

			{
				osg::StateSet* ss = hudTextGeode->getOrCreateStateSet();
				osg::ref_ptr<osg::Program> shaderProgram = new osg::Program;

				osg::ref_ptr<osg::Shader> vs = new osg::Shader(osg::Shader::VERTEX);
				osg::ref_ptr<osg::Shader> fs = new osg::Shader(osg::Shader::FRAGMENT);

				vs->loadShaderSourceFromFile("media/hudText.vs");
				fs->loadShaderSourceFromFile("media/hudText.fs");

				shaderProgram->addShader(vs);
				shaderProgram->addShader(fs);

				ss->addUniform(new osg::Uniform("texture_Diffuse", 0));

				ss->setAttributeAndModes(shaderProgram, osg::StateAttribute::ON);
			}
		}

		{
			osg::StateSet* ss = mvMatrix->getOrCreateStateSet();
			ss->setMode(GL_BLEND, osg::StateAttribute::ON);
			ss->setMode(GL_DEPTH_TEST, osg::StateAttribute::OFF);
			ss->setRenderingHint(osg::StateSet::TRANSPARENT_BIN);
			ss->setRenderBinDetails(11, "RenderBin");
		}
	}
	// End HUD


	// Set up debug grid.
	{
		osg::Geode* debugGeode = new osg::Geode();
		osg::StateSet* ss = debugGeode->getOrCreateStateSet();
		ss->setAttributeAndModes(new osg::LineWidth(4.0f));
		ss->setMode(GL_BLEND, osg::StateAttribute::ON);
		ss->setRenderingHint(osg::StateSet::TRANSPARENT_BIN);

		{
			osg::ref_ptr<osg::Program> shaderProgram = new osg::Program;

			osg::ref_ptr<osg::Shader> vs = new osg::Shader(osg::Shader::VERTEX);
			osg::ref_ptr<osg::Shader> fs = new osg::Shader(osg::Shader::FRAGMENT);

			vs->loadShaderSourceFromFile("media/debugGrid.vs");
			fs->loadShaderSourceFromFile("media/debugGrid.fs");

			shaderProgram->addShader(vs);
			shaderProgram->addShader(fs);

			ss->setAttributeAndModes(shaderProgram, osg::StateAttribute::ON);
		}


		osg::Geometry* debugGeom = new osg::Geometry();

		CalculateBoundingBoxVisitor cbbv;
		scene->accept(cbbv);
		const osg::BoundingBox bb = cbbv.getBoundBox();

		int r = LPV_SIZE;

		osg::Vec3f gridSize = (bb._max - bb._min) / r;

		osg::Vec3Array* vertices = new osg::Vec3Array;
		for(int i = 0; i <= r; i++) {
			for(int j = 0; j <= r; j++) {
				vertices->push_back(osg::componentMultiply(osg::Vec3f(i,j,0), gridSize) + bb._min);
				vertices->push_back(osg::componentMultiply(osg::Vec3f(i,j,r), gridSize) + bb._min);

				vertices->push_back(osg::componentMultiply(osg::Vec3f(i,0,j), gridSize) + bb._min);
				vertices->push_back(osg::componentMultiply(osg::Vec3f(i,r,j), gridSize) + bb._min);

				vertices->push_back(osg::componentMultiply(osg::Vec3f(0,i,j), gridSize) + bb._min);
				vertices->push_back(osg::componentMultiply(osg::Vec3f(r,i,j), gridSize) + bb._min);
			}
		}
		debugGeom->setVertexArray(vertices);

		osg::Vec4Array* colors = new osg::Vec4Array;
		colors->push_back(osg::Vec4(1.0f,0.0f,0.0f,0.5f));
		debugGeom->setColorArray(colors);
		debugGeom->setColorBinding(osg::Geometry::BIND_OVERALL);

		debugGeom->addPrimitiveSet(new osg::DrawArrays(osg::PrimitiveSet::LINES, 0, vertices->size()));

		debugGeode->addDrawable(debugGeom);
		rootNode->addChild(debugGeode);

		config->setDebugNode(debugGeode);
	}

	// Set up debug SH spheres.
	{
		osg::Geode* debugGeode = new osg::Geode();
		osg::StateSet* ss = debugGeode->getOrCreateStateSet();

		{
			osg::ref_ptr<osg::Program> shaderProgram = new osg::Program;

			osg::ref_ptr<osg::Shader> vs = new osg::Shader(osg::Shader::VERTEX);
			osg::ref_ptr<osg::Shader> fs = new osg::Shader(osg::Shader::FRAGMENT);

			vs->loadShaderSourceFromFile("media/debugSH.vs");
			fs->loadShaderSourceFromFile("media/debugSH.fs");

			shaderProgram->addShader(vs);
			shaderProgram->addShader(fs);

			ss->setAttributeAndModes(shaderProgram, osg::StateAttribute::ON);

			osg::Uniform* lpvUniform = new osg::Uniform(osg::Uniform::SAMPLER_3D, "volume_LightPropagation", 3);

			data::LPVComponents* accumulationBuffer = lpv->getAccumulationBuffer();

			ss->setTextureAttributeAndModes(TEXTURE_ID_LPV_START+0, accumulationBuffer->getRed()->getTexture(), osg::StateAttribute::ON);
			ss->setTextureAttributeAndModes(TEXTURE_ID_LPV_START+1, accumulationBuffer->getGreen()->getTexture(), osg::StateAttribute::ON);
			ss->setTextureAttributeAndModes(TEXTURE_ID_LPV_START+2, accumulationBuffer->getBlue()->getTexture(), osg::StateAttribute::ON);
			lpvUniform->setElement(0, TEXTURE_ID_LPV_START+0);
			lpvUniform->setElement(1, TEXTURE_ID_LPV_START+1);
			lpvUniform->setElement(2, TEXTURE_ID_LPV_START+2);

			ss->addUniform(lpvUniform);
		}


		CalculateBoundingBoxVisitor cbbv;
		scene->accept(cbbv);
		const osg::BoundingBox bb = cbbv.getBoundBox();

		int r = LPV_SIZE;

		osg::Vec3f gridSize = (bb._max - bb._min) / r;

		osg::Vec3Array* vertices = new osg::Vec3Array;
		for(int i = 0; i < r; i++) {
			for(int j = 0; j < r; j++) {
				for(int k = 0; k < r; k++) {
					osg::Vec3f pos = osg::componentMultiply(osg::Vec3f(i,j,k), gridSize) + bb._min + gridSize * 0.5f;
					debugGeode->addDrawable(new osg::ShapeDrawable(new osg::Box(pos, gridSize.x()/5.0f, gridSize.y()/5.0f, gridSize.z()/5.0f)));
				}
			}
		}
		rootNode->addChild(debugGeode);
		config->setDebugSHNode(debugGeode);
	}


	std::cout << "Setting up and starting viewer." << std::endl;

	/*
	 * Must run in single threaded mode for osgCompute/osgCuda computations to be processed correctly.
	 */
	viewer.setThreadingModel(osgViewer::Viewer::SingleThreaded);

	viewer.setSceneData(rootNode);

	//viewer.setUpViewInWindow(0, 0, 1280, 800, 0);

	viewer.realize();

	/*
	 * Tell OSG not to adjust the near and far plane since that would break
	 * the alignment between the different sides in the shadow cubemap.
	 */
	viewer.getCamera()->setComputeNearFarMode(osgUtil::CullVisitor::DO_NOT_COMPUTE_NEAR_FAR);

	/*
	 * Tell OSG to provide us with its own osg_-prefixed transformation uniforms and vertex attributes.
	 */
	viewer.getCamera()->getGraphicsContext()->getState()->setUseModelViewAndProjectionUniforms(
			true);
	viewer.getCamera()->getGraphicsContext()->getState()->setUseVertexAttributeAliasing(
			true);
	cameraManipulator->setByMatrix(osg::Matrixd::identity());
	viewer.run();

	TimeMonitor::instance()->dump("User shutdown");
}
