package rcon;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Properties;

public class DaemonConfiguration {
	
	private Properties	config;
	
	public DaemonConfiguration(String filename) throws IOException {
		config = new Properties();
		File f = new File(filename);
		if(f.exists()) {
			FileInputStream in = new FileInputStream(filename);
			config.load(in);
			in.close();
		} else {
			throw new IOException("No such file.");
		}
	}
	
	public String getServerHost() {
		return getString("server.host");
	}
	
	public int getServerPort() {
		return getInt("server.port");
	}
	
	public String getServerPassword() {
		return getString("server.password");
	}
	
	public int getServerReconnectDelay() {
		return getInt("server.reconnectDelay", -1);
	}
	
	public int getServerTimeout() {
		return getInt("server.timeout", 5000);
	}
	
	public String getServerConfig() {
		return getString("server.config", "config.properties");
	}
	
	public boolean getServerConfigAutoApply() {
		return getBoolean("server.config.autoapply", false);
	}
	
	public String getDatabaseDriver() {
		return getString("database.driver");
	}
	
	public String getDatabaseUrl() {
		return getString("database.url");
	}
	
	public String getDatabaseUser() {
		return getString("database.user");
	}
	
	public String getDatabasePassword() {
		return getString("database.password");
	}
	
	public boolean isHeartbeatsEnabled() {
		return getBoolean("heartbeat.enabled", false);
	}
	
	public int getHeartbeatInterval() {
		return getInt("heartbeat.interval", 30000);
	}
	
	public boolean getHttpdEnabled() {
		return getBoolean("httpd.enabled", false);
	}
	
	public int getHttpdPort() {
		return getInt("httpd.port", 5391);
	}
	
	public boolean getClientsEnabled() {
		return getBoolean("clients.enabled", false);
	}
	
	public int getClientsPort() {
		return getInt("clients.port", 5390);
	}
	
	public String getTimeZone() {
		return getString("daemon.timeZone", "GMT");
	}
	
	public String getDateFormat() {
		return getString("daemon.dateFormat", "MMMM d, yyyy HH:mm:ss zzz");
	}
	
	public String getDaemonId() {
		return getString("daemon.id", "tgac-main");
	}
	
	public String getBannedMessage() {
		return getString("ban.message", "You are banned until %1$s for %2$s.");
	}
	
	public int getPlayerUpdateInterval() {
		return getInt("players.updateInterval", 30000);
	}
	
	public boolean isServerMessageEnabled() {
		return getBoolean("messages.enabled", true);
	}
	
	public int getServerMessageInterval() {
		return getInt("messages.interval", 60000);
	}
	
	public String getServerMessageFile() {
		return getString("messages.file", "messages.txt");
	}
	
	public boolean isKillStreaksEnabled() {
		return getBoolean("streaks.enabled", true);
	}
	
	public int getStreakModulus() {
		return getInt("streaks.modulus", 5);
	}
	
	public int getStreakAwesomeLimit() {
		return getInt("streaks.awesome", 15);
	}
	
	public boolean isAutoConfigEnabled() {
		return getBoolean("autoconfig.enabled", false);
	}
	
	public String getWelcomeMessage() {
		return getString("server.welcome", "");
	}
	
	public boolean isKDNotificationsEnabled() {
		return getBoolean("kdnotify.enabled", true);
	}
	
	public double getKDRatio() {
		return getDouble("kdnotify.ratio", 5);
	}
	
	public int getKDMinKills() {
		return getInt("kdnotify.kills", 20);
	}
	
	private int getInt(String name) {
		try {
			return Integer.parseInt(getString(name));
		} catch(NumberFormatException e) {
			throw new ConfigurationException("Expected numeric value for " + name + ".", e);
		}
	}
	
	private double getDouble(String name, int i) {
		try {
			return Double.parseDouble(getString(name));
		} catch(NumberFormatException e) {
			throw new ConfigurationException("Expected numeric value for " + name + ".", e);
		}
	}
	
	private String getString(String name) {
		String ret = config.getProperty(name);
		if(ret == null) {
			throw new ConfigurationException("Missing configuration value for " + name + ".");
		} else {
			return ret;
		}
	}
	
	private String getString(String name, String def) {
		return config.getProperty(name, def);
	}
	
	private int getInt(String name, int def) {
		try {
			return Integer.parseInt(getString(name, String.valueOf(def)));
		} catch(NumberFormatException e) {
			throw new ConfigurationException("Expected numeric value for " + name + ".", e);
		}
	}
	
	private boolean getBoolean(String name) {
		String value = getString(name);
		if(value.equalsIgnoreCase("true") || value.equalsIgnoreCase("yes")) {
			return true;
		} else if(value.equalsIgnoreCase("false") || value.equalsIgnoreCase("no")) {
			return false;
		} else {
			throw new ConfigurationException("Expected boolean value (true or false) for " + name + ".");
		}
	}
	
	private boolean getBoolean(String name, boolean def) {
		String value = getString(name, String.valueOf(def));
		if(value.equalsIgnoreCase("true") || value.equalsIgnoreCase("yes")) {
			return true;
		} else if(value.equalsIgnoreCase("false") || value.equalsIgnoreCase("no")) {
			return false;
		} else {
			throw new ConfigurationException("Expected boolean value (true or false) for " + name + ".");
		}
	}
	
	public int getServerAdminId() {
		return getInt("server.admin.id", 0);
	}
	
}
