package rcon;

import java.io.IOException;
import java.sql.SQLException;
import java.util.Calendar;
import java.util.EnumSet;

import org.apache.log4j.Logger;

import rcon.client.ClientHandler;
import rcon.database.Database;
import rcon.event.AbandonReportHandler;
import rcon.event.AutomaticIdleHandler;
import rcon.event.BanEnforcerHandler;
import rcon.event.DatabaseSynchronizationHandler;
import rcon.event.KDNotificationHandler;
import rcon.event.StatsHandler;
import rcon.event.StreakHandler;
import rcon.event.WelcomeHandler;
import rcon.ingame.InGameHandler;
import rcon.ingame.ServerMessages;
import rcon.log.ChatLogHandler;
import rcon.players.Admin;
import rcon.players.Player;
import rcon.players.Right;
import rcon.players.ServerPlayer;
import rcon.server.GameServer;

public class RCONDaemon {
	
	static Logger				logger	= Logger.getLogger(RCONDaemon.class);
	
	private DaemonConfiguration	config;
	private Database			db;
	private GameServer			server;
	
	private volatile boolean	running;
	
	private ClientHandler		clients;
	
	private long				started;
	
	public long getStarted() {
		return started;
	}
	
	public RCONDaemon(DaemonConfiguration config) {
		this.config = config;
		this.db = null;
		this.server = null;
		running = false;
	}
	
	public boolean init() {
		try {
			db = new Database(config.getDaemonId(), config.getDatabaseDriver(), config.getDatabaseUrl(),
				config.getDatabaseUser(), config.getDatabasePassword());
			return true;
		} catch(SQLException e) {
			logger.error("Failed to establish database connection. ", e);
			return false;
		}
	}
	
	public void start() {
		running = true;
		started = Calendar.getInstance().getTimeInMillis() / 1000;
		if(config.getClientsEnabled()) {
			clients = new ClientHandler(this, config.getClientsPort());
			clients.start();
		}
		
		do {
			server = null;
			try {
				server = new GameServer(config.getServerHost(), config.getServerPort(), config.getServerTimeout(),
					config.getServerConfig());
			} catch(IOException e) {
				logger.error("Failed to connect to game server.", e);
				server = null;
			}
			if(server != null) {
				int serverAdminId = config.getServerAdminId();
				System.out.println("Server admin id is " + serverAdminId + ".");
				if(serverAdminId != 0) {
					Player serverPlayer = new ServerPlayer("Server");
					serverPlayer.setAdmin(new Admin(serverAdminId, "", "Server", "", EnumSet.allOf(Right.class)));
					System.out.println("Set the server admin player!");
					server.getOnline().setServerPlayer(serverPlayer);
				}
				if(server.login(config.getServerPassword()) && server.enableEvents()) {
					
					try {
						db.logs.event("Connected to game server at " + server.getHost() + ":" + server.getPort() + ".");
					} catch(SQLException e1) {
						e1.printStackTrace();
					}
					
					server.addHandler(new DatabaseSynchronizationHandler(db));
					server.addHandler(new InGameHandler(server, db));
					server.addHandler(new BanEnforcerHandler(server, db, config.getDateFormat(), config.getTimeZone(),
						config.getBannedMessage()));
					server.addHandler(new ChatLogHandler(db));
					server.addHandler(new StatsHandler(server, db));
					server.addHandler(new AbandonReportHandler(db));
					
					if(config.isKillStreaksEnabled()) {
						int modulus = config.getStreakModulus();
						int bigLimit = config.getStreakAwesomeLimit();
						server.addHandler(new StreakHandler(server, modulus, bigLimit));
					}
					if(config.isAutoConfigEnabled()) {
						server.addHandler(new AutomaticIdleHandler(server));
					}
					if(!config.getWelcomeMessage().trim().isEmpty()) {
						server.addHandler(new WelcomeHandler(server, config.getWelcomeMessage()));
					}
					
					if(config.isKDNotificationsEnabled()) {
						double kd = config.getKDRatio();
						int minKills = config.getKDMinKills();
						server.addHandler(new KDNotificationHandler(server, kd, minKills));
					}
					
					/*
					 * if(config.getServerConfigAutoApply()) {
					 * server.applyConfiguration(); }
					 */
					
					server.requestPlayerInfo();
					server.updateInfo();
					
					server.enablePlayerUpdates(config.getPlayerUpdateInterval());
					
					server.requestPbPlayerList();
					
					if(config.isHeartbeatsEnabled()) {
						server.enableHeartbeats(config.getHeartbeatInterval());
					}
					
					if(config.isServerMessageEnabled()) {
						ServerMessages messages = server.getServerMessages();
						messages.setFile(config.getServerMessageFile());
						try {
							messages.load();
						} catch(IOException e) {
							logger.error("Failed to load server messages.", e);
						}
						server.enableServerMessenger(config.getServerMessageInterval());
					}
				}
				
				for(Player p : server.getOnline().getPlayers()) {
					p.spawn();
				}
				
				server.awaitDisconnect();
				try {
					db.logs.event("Lost connection to game server with " + server.getOnline().size()
							+ " players online.");
				} catch(SQLException e) {
					e.printStackTrace();
				}
				server = null;
			}
			if(running && config.getServerReconnectDelay() != -1) {
				try {
					Thread.sleep(config.getServerReconnectDelay());
				} catch(InterruptedException e) {}
			}
			
		} while(running && config.getServerReconnectDelay() != -1);
		
		if(clients != null) {
			clients.close();
			clients = null;
		}
	}
	
	public GameServer getServer() {
		return server;
	}
	
	public Database getDatabase() {
		return db;
	}
	
}
