package rcon.client;

import java.io.IOException;
import java.net.Socket;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.sql.SQLException;
import java.util.EnumSet;
import java.util.Iterator;

import rcon.RCONDaemon;
import rcon.Hex;
import rcon.MessageType;
import rcon.commands.Command;
import rcon.net.RCONInputStream;
import rcon.net.RCONOutputStream;
import rcon.players.Admin;
import rcon.players.Right;
import rcon.server.GameServer;

public class ClientConnection extends Thread {
	
	private static final int	SEED_LENGTH	= 16;
	private static SecureRandom	random;
	static {
		try {
			random = SecureRandom.getInstance("SHA1PRNG");
		} catch(NoSuchAlgorithmException e) {
			e.printStackTrace();
		}
	}
	
	private Socket				socket;
	private Admin				user;
	private byte[]				salt;
	private RCONInputStream		in;
	private RCONOutputStream	out;
	private RCONDaemon				daemon;
	private ClientHandler		clients;
	
	private BanModule			bans;
	private AdminModule			admins;
	private ConfigModule		config;
	private PlayerModule		players;
	private PunkbusterModule	punkbuster;
	private MessageModule		messages;
	private LogModule			logs;
	private ReservedSlotsModule	reservedSlots;
	
	public ClientConnection(Socket socket, ClientHandler clients, RCONDaemon daemon) throws IOException {
		this.socket = socket;
		this.in = new RCONInputStream(socket.getInputStream());
		this.out = new RCONOutputStream(socket.getOutputStream());
		this.clients = clients;
		this.daemon = daemon;
		this.user = null;
		this.salt = null;
		
		this.bans = new BanModule(out, daemon);
		this.admins = new AdminModule(out, daemon);
		this.config = new ConfigModule(out, daemon);
		this.players = new PlayerModule(out, daemon);
		this.punkbuster = new PunkbusterModule(out, daemon);
		this.messages = new MessageModule(out, daemon);
		this.logs = new LogModule(out, daemon);
		this.reservedSlots = new ReservedSlotsModule(out, daemon);
	}
	
	@Override
	public void run() {
		try {
			while(socket.isConnected()) {
				Command request = in.readCommand();
				
				ClientCommand command = ClientCommand.get(request.getName());
				
				EnumSet<Right> rights = user == null ? EnumSet.noneOf(Right.class) : user.getRights();
				
				if(daemon.getServer() != null && command != null && command.hasRights(rights)) {
					switch(command) {
						/* Default module */
						case VERSION:
							parseVersion(user, request);
							break;
						case QUIT:
							parseQuit(user, request);
							break;
						case LOGIN_HASHED:
							parseLogin(user, request);
							break;
						case LOGOUT:
							parseLogout(user, request);
							break;
						case SHUTDOWN:
							parseShutdown(user, request);
							break;
						
						case DAEMON_STATUS:
							daemonStatus(user, request);
							break;
						case SERVER_STATUS:
							serverStatus(user, request);
							break;
						
						/* Configuration module */
						case CONFIG_SET:
							config.set(user, request);
							break;
						case CONFIG_GET:
							config.get(user, request);
							break;
						case CONFIG_CLEAR:
							config.clear(user, request);
							break;
						case CONFIG_LIST:
							config.list(user, request);
							break;
						case CONFIG_ALL:
							config.all(user, request);
							break;
						
						/* Bans module */
						case BANS_LIST:
							bans.list(user, request);
							break;
						case BANS_HISTORY:
							bans.history(user, request);
							break;
						case BANS_REMOVE:
							bans.remove(user, request);
							break;
						case BANS_BAN:
							bans.ban(user, request);
							break;
						case BANS_KICK:
							bans.kick(user, request);
							break;
						case BANS_EDIT:
							bans.edit(user, request);
							break;
						case BANS_GET:
							bans.get(user, request);
							break;
						case BANS_WARN:
							bans.warn(user, request);
							break;
						
						/* Admins module */
						case ADMINS_SET_RIGHTS:
							admins.setRights(user, request);
							break;
						case ADMINS_GET:
							admins.get(user, request);
							break;
						case ADMINS_SET_PASSWORD:
							admins.setPassword(user, request);
							break;
						case ADMINS_REGISTER:
							admins.register(user, request);
							break;
						case ADMINS_EDIT:
							admins.edit(user, request);
							break;
						case ADMINS_LIST:
							admins.list(user, request);
							break;
						
						/* Players module */
						case PLAYERS_LIST:
							players.list(user, request);
							break;
						case PLAYERS_SEARCH:
							players.search(user, request);
							break;
						case PLAYERS_SAY:
							players.say(user, request);
							break;
						case PLAYERS_YELL:
							players.yell(user, request);
							break;
						
						/* Punkbuster module */
						case PUNKBUSTER_EXECUTE:
							punkbuster.execute(user, request);
							break;
						
						/* Messages module */
						case MESSAGES_LIST:
							messages.list(user, request);
							break;
						case MESSAGES_ADD:
							messages.add(user, request);
							break;
						case MESSAGES_EDIT:
							messages.edit(user, request);
							break;
						case MESSAGES_REMOVE:
							messages.remove(user, request);
							break;
						
						/* Logs module */
						case LOGS_ADMIN:
							logs.admin(user, request);
							break;
						case LOGS_CHAT:
							logs.chat(user, request);
							break;
						case LOGS_EVENT:
							logs.event(user, request);
							break;
						
						/* Reserved slots */
						case RESERVEDSLOTS_LOAD:
							reservedSlots.load(user, request);
							break;
						case RESERVEDSLOTS_SAVE:
							reservedSlots.save(user, request);
							break;
						case RESERVEDSLOTS_LIST:
							reservedSlots.list(user, request);
							break;
						case RESERVEDSLOTS_CLEAR:
							reservedSlots.clear(user, request);
							break;
						case RESERVEDSLOTS_ADDPLAYER:
							reservedSlots.addPlayer(user, request);
							break;
						case RESERVEDSLOTS_REMOVEPLAYER:
							reservedSlots.removePlayer(user, request);
							break;
						
						default:
							reply(request, "InvalidCommand");
							break;
					}
				} else if(daemon.getServer() == null) {
					reply(request, "NotConnected");
				} else if(command == null) {
					reply(request, "InvalidCommand");
				} else {
					reply(request, "NotAuthorized");
				}
			}
		} catch(IOException e) {
			System.out.println("Client connection closed.");
		} finally {
			clients.onClientDisconnect(this);
		}
	}
	
	private void serverStatus(Admin user2, Command request) throws IOException {
		if(daemon.getServer() == null) {
			reply(request, "ServerOffline");
		} else {
			GameServer server = daemon.getServer();
			String host = server.getHost();
			String port = String.valueOf(server.getPort());
			String mode = server.getGameMode();
			String map = server.getMap();
			String online = String.valueOf(server.getOnline().size());
			String max = String.valueOf(server.getMaxPlayerCount());
			String round = String.valueOf(server.getRoundsPlayed());
			String rounds = String.valueOf(server.getRoundsTotal());
			reply(request, "OK", host, port, mode, map, online, max, round, rounds);
		}
	}
	
	private void daemonStatus(Admin user2, Command request) throws IOException {
		String start = String.valueOf(daemon.getStarted());
		String connect = "-1";
		if(daemon.getServer() != null) {
			connect = String.valueOf(daemon.getServer().getConnectionTime());
		}
		String clients = String.valueOf(this.clients.getClientCount());
		reply(request, "OK", start, connect, clients);
	}
	
	private void parseVersion(Admin user2, Command request) throws IOException {
		reply(request, "OK", "TGAC", "0.1");
	}
	
	private void parseLogin(Admin user2, Command request) throws IOException {
		if(request.size() == 0) {
			salt = new byte[SEED_LENGTH];
			random.nextBytes(salt);
			reply(request, "OK", Hex.toHex(salt));
		} else if(request.size() == 2) {
			if(salt == null) {
				reply(request, "MissingSalt");
			} else {
				String username = request.get(0);
				String passwordHashHex = request.get(1);
				try {
					Admin user = daemon.getDatabase().admins.get(username);
					if(user == null) {
						reply(request, "InvalidLogin");
					} else {
						String realPasswordHashHex = Hex.hashPassword(user.getPassword(), Hex.toHex(salt));
						if(passwordHashHex.equalsIgnoreCase(realPasswordHashHex)) {
							if(user.hasRight(Right.console)) {
								this.user = user;
								EnumSet<Right> rights = user.getRights();
								String[] arguments = new String[rights.size() + 1];
								Iterator<Right> it = rights.iterator();
								arguments[0] = String.valueOf(rights.size());
								for(int i = 1; i <= rights.size(); i++) {
									arguments[i] = it.next().name();
								}
								daemon.getDatabase().logs.admin(user, "Logged in on console.");
								reply(request, "OK", arguments);
							} else {
								reply(request, "NotAuthorized");
							}
						} else {
							reply(request, "InvalidLogin");
						}
					}
				} catch(SQLException e) {
					reply(request, "ServerError");
					e.printStackTrace();
				}
				salt = null;
			}
		} else {
			reply(request, "InvalidArguments");
			salt = null;
		}
	}
	
	private void parseLogout(Admin user2, Command request) throws IOException {
		user = null;
		reply(request, "OK");
	}
	
	private void parseQuit(Admin user2, Command request) throws IOException {
		reply(request, "OK");
		socket.close();
	}
	
	private void parseShutdown(Admin user2, Command request) throws IOException {
		reply(request, "OK");
		socket.close();
	}
	
	private void reply(Command request, String reply, String... arguments) throws IOException {
		out.sendCommand(new Command(reply, MessageType.client_response, request.getSeq(), arguments));
	}
	
	public void close() {
		try {
			socket.close();
		} catch(IOException e) {} finally {
			try {
				join(1000);
			} catch(InterruptedException e) {}
			if(isAlive()) {
				clients.onClientDisconnect(this);
			}
		}
	}
	
}
