package rcon.ingame;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.regex.Matcher;

import org.apache.log4j.Logger;

import rcon.Scheduler;
import rcon.database.Database;
import rcon.event.BanDuration;
import rcon.event.RCONEventAdapter;
import rcon.players.Player;
import rcon.players.Punishment;
import rcon.players.Ranking;
import rcon.players.Report;
import rcon.players.Right;
import rcon.server.GameServer;

public class InGameHandler extends RCONEventAdapter {
	
	static Logger		logger	= Logger.getLogger(InGameHandler.class);
	
	private Database	db;
	private GameServer	server;
	
	public InGameHandler(GameServer server, Database db) {
		this.server = server;
		this.db = db;
	}
	
	private void log(Player user, String message) {
		try {
			db.logs.admin(user.getAdmin(), message);
		} catch(SQLException e) {
			e.printStackTrace();
		}
	}
	
	@Override
	public void onChat(Player player, String message) {
		if(message.length() >= 1 && message.charAt(0) == '/') {
			message = message.substring(1);
		}
		if(message.length() >= 1 && message.charAt(0) == '!') {
			InGameCommand command = InGameCommand.get(message);
			if(command != null) {
				EnumSet<Right> rights = EnumSet.of(Right.report, Right.rank);
				if(player.isAdmin()) {
					rights.addAll(player.getAdmin().getRights());
				}
				if(command.hasRights(rights)) {
					Matcher m = command.match(message);
					if(m.matches()) {
						switch(command) {
							case nextMap:
								parseNextMap(player, m);
								break;
							case restart:
								parseRestart(player, m);
								break;
							case whisper:
								parseWhisper(player, m);
								break;
							case warn:
								parseWarnCommand(player, m);
								break;
							case kick:
								parseKickCommand(player, m);
								break;
							case ban:
								parseBanCommand(player, m);
								break;
							case kill:
								parseKillCommand(player, m);
								break;
							case yell:
								parseYellCommand(player, m);
								break;
							case adminChat:
								parseAdminChatCommand(player, m);
								break;
							case history:
								parseHistoryCommand(player, m);
								break;
							case report:
								parseReport(player, m);
								break;
							case rank:
								parseRank(player, m);
								break;
							case pb:
								parsePunkbusterCommand(player, m);
								break;
						}
					} else {
						server.say("Invalid syntax. " + command.getSyntax(), player);
					}
				}
			}
			
			if(player.isAdmin()) {
				logger.trace("Admin player issued command " + message + ". " + player.toString() + ", "
						+ player.getAdmin().toString() + ". (" + command + ")");
			} else {
				logger.trace("Non-admin player issued command " + message + ". " + player.toString() + ". (" + command
						+ ")");
			}
		}
	}
	
	private void parsePunkbusterCommand(Player player, Matcher m) {
		String command = m.group(1);
		server.punkbuster(command);
		log(player, "Executed punkbuster command: \"" + command + "\".");
	}
	
	private void parseKillCommand(Player player, Matcher m) {
		String name = m.group(1);
		Player target = getPlayerByPartialName(player, name);
		if(target != null) {
			name = target.getName();
			
			server.kill(target);
			log(player, "Killed " + target.getName() + ".");
		}
	}
	
	private void parseWhisper(Player player, Matcher matcher) {
		String name = matcher.group(1);
		String message = matcher.group(2);
		
		Player target = getPlayerByPartialName(player, name);
		if(target != null) {
			name = target.getName();
			
			String senderMessage = "To " + name + ": " + message;
			String targetMessage = "Admin: " + message;
			
			server.say(senderMessage, player);
			server.say(targetMessage, target);
			log(player, "Whispered '" + message + "' to '" + target.getName() + "'.");
		}
	}
	
	private void parseReport(Player player, Matcher m) {
		String text = m.group(1);
		ArrayList<Player> admins = server.getOnline().getAdmins();
		if(text.trim().isEmpty()) {
			server.say("Please include a short description of your report. (!report <description>)", player);
		} else {
			try {
				if(player.getActiveReport() == null) {
					String serverName = server.getServerName();
					StringBuilder description = new StringBuilder();
					description.append("Report on: " + serverName + " (" + server.getHost() + ":" + server.getPort()
							+ ")\n");
					description.append(text);
					Report r = db.reports.submitReport(player, text, description.toString());
					if(r == null) {
						server.say("Your report could not be submitted due to an internal error.", player);
						server.say("Please submit the report on our website.", player);
					} else {
						server.say("Thank you. Please be patient while our admins review your report.", player);
						server.say("You can add more information to your report with the same command.", player);
						player.setActiveReport(r);
					}
				} else {
					Report r = player.getActiveReport();
					db.reports.comment(r, text);
					server.say("Your comment has been added to your pending report.", player);
				}
			} catch(SQLException e) {
				e.printStackTrace();
			}
			
			for(Player p : admins) {
				server.say(player.getName() + " reports: " + text, p);
			}
		}
	}
	
	private void parseRank(Player player, Matcher m) {
		try {
			Ranking rank = db.stats.getRank(player);
			server.say("You are ranked " + rank.rank + " of " + rank.amount + " based on your score.", player);
		} catch(SQLException e) {
			e.printStackTrace();
		}
	}
	
	private void parseRestart(Player player, Matcher m) {
		server.say("Restarting map in ten seconds.");
		log(player, "Restarted the map.");
		
		Scheduler.getInstance().schedule(new Runnable() {
			@Override
			public void run() {
				server.restartMap();
			}
		}, 10000);
	}
	
	private void parseNextMap(Player player, Matcher m) {
		server.say("Changing map in ten seconds.");
		log(player, "Changed to next map.");
		
		Scheduler.getInstance().schedule(new Runnable() {
			@Override
			public void run() {
				server.runNextMap();
			}
		}, 10000);
	}
	
	private void parseHistoryCommand(Player player, Matcher m) {
		String name = m.group(1);
		Player target = getPlayerByPartialName(player, name);
		if(target != null) {
			try {
				ArrayList<Punishment> punishments = db.bans.getByPlayer(target);
				if(punishments.isEmpty()) {
					server.say("Player " + target.getName() + " has no history of rule violations.", player);
				} else {
					int warns = 0;
					int kicks = 0;
					int bans = 0;
					for(Punishment p : punishments) {
						if(p.isEnabled()) {
							switch(p.getType()) {
								case warn:
									warns++;
									break;
								case kick:
									kicks++;
									break;
								case ban:
									bans++;
									break;
							}
						}
					}
					StringBuilder sb = new StringBuilder();
					sb.append("Player ");
					sb.append(target.getName());
					sb.append(" has been ");
					if(warns > 0) {
						sb.append("warned ");
						if(warns == 1) {
							sb.append("one time");
						} else {
							sb.append(warns);
							sb.append(" times");
						}
						if(kicks > 0 && bans > 0) {
							sb.append(", ");
						} else if(kicks > 0 || bans > 0) {
							sb.append(" and ");
						} else {
							sb.append('.');
						}
					}
					if(kicks > 0) {
						sb.append("kicked ");
						if(kicks == 1) {
							sb.append("one time");
						} else {
							sb.append(kicks);
							sb.append(" times");
						}
						if(bans > 0) {
							sb.append(" and ");
						} else {
							sb.append('.');
						}
					}
					if(bans > 0) {
						sb.append("banned ");
						if(bans == 1) {
							sb.append("one time.");
						} else {
							sb.append(bans);
							sb.append(" times.");
						}
					}
					server.say(sb.toString(), player);
				}
				
			} catch(SQLException e) {
				logger.error("Database exception while invoking history command.", e);
			}
		}
	}
	
	private void parseAdminChatCommand(Player player, Matcher matcher) {
		String text = matcher.group(1);
		ArrayList<Player> admins = server.getOnline().getAdmins();
		if(admins.size() <= 1) {
			server.say("There are no other admins online.", player);
		} else {
			for(Player p : admins) {
				server.say(player.getName() + ": " + text, p);
				log(player, "Said '" + text + "' to the other admins.");
			}
		}
	}
	
	private void parseBanCommand(Player player, Matcher matcher) {
		String name = matcher.group(1);
		String reason = matcher.group(2);
		String time = matcher.group(3);
		
		final Player target = getPlayerByPartialName(player, name);
		if(target != null) {
			name = target.getName();
			long timeout = parseTimeout(time);
			
			if(timeout == -1) {
				server.say(time + " is not a valid ban time. "
						+ "Expected one of: moment, hour, day, week, month or perm.", player);
			} else {
				try {
					db.bans.add(Punishment.ban(target, timeout, reason, player.getAdmin()));
					reason = db.bans.getReason(reason.trim());
				} catch(SQLException e) {
					logger.error("Database exception while invoking ban command.", e);
				}
				server.ban(target, reason, time);
				log(player, "Banned '" + target.getName() + "' (" + target.getEaGuid() + ", " + target.getPbGuid()
						+ ") for " + reason + ". Duration: " + time + ".");
			}
		}
	}
	
	private void parseKickCommand(Player player, Matcher matcher) {
		String name = matcher.group(1);
		String reason = matcher.group(2);
		
		Player target = getPlayerByPartialName(player, name);
		if(target != null) {
			name = target.getName();
			try {
				db.bans.add(Punishment.kick(target, reason, player.getAdmin()));
				reason = db.bans.getReason(reason.trim());
			} catch(SQLException e) {
				logger.error("Database exception while invoking kick command.", e);
			}
			server.kick(target, reason);
			log(player, "Kicked '" + target.getName() + "' (" + target.getEaGuid() + ", " + target.getPbGuid()
					+ ") for " + reason + ".");
		}
	}
	
	private void parseWarnCommand(Player player, Matcher matcher) {
		String name = matcher.group(1);
		String reason = matcher.group(2);
		
		Player target = getPlayerByPartialName(player, name);
		if(target != null) {
			name = target.getName();
			
			try {
				db.bans.add(Punishment.warn(target, reason, player.getAdmin()));
				reason = db.bans.getReason(reason.trim());
			} catch(SQLException e) {
				logger.error("Database exception while invoking warn command.", e);
			}
			server.warn(target, reason);
			log(player, "Warned '" + target.getName() + "' (" + target.getEaGuid() + ", " + target.getPbGuid()
					+ ") for " + reason + ".");
		}
	}
	
	private void parseYellCommand(Player player, Matcher matcher) {
		String text = matcher.group(1);
		server.say(text);
		log(player, "Yelled '" + text + "' to everyone.");
	}
	
	private Player getPlayerByPartialName(Player issuer, String part) {
		ArrayList<Player> matches = server.getOnline().getByPartialName(part);
		if(matches.size() == 1) {
			return matches.get(0);
		} else if(matches.size() > 1) {
			server.say(ambiguousNameString(matches), issuer);
		} else if(matches.isEmpty()) {
			server.say("No match for player name " + part + ".", issuer);
		}
		return null;
	}
	
	private String ambiguousNameString(ArrayList<Player> matches) {
		StringBuilder sb = new StringBuilder();
		sb.append("Ambiguous name. ");
		sb.append(matches.size());
		sb.append(" possible matches: ");
		Iterator<Player> it = matches.iterator();
		while(it.hasNext()) {
			Player p = it.next();
			sb.append(p.getFullName());
			if(it.hasNext()) {
				sb.append(", ");
			}
		}
		sb.append(".");
		String outMessage = sb.toString();
		return outMessage;
	}
	
	private long parseTimeout(String time) {
		try {
			BanDuration duration = Enum.valueOf(BanDuration.class, time.trim().toLowerCase());
			return duration.getDuration();
		} catch(IllegalArgumentException e) {
			return -1;
		}
	}
	
}
