package rcon.ingame;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.ListIterator;
import java.util.NoSuchElementException;

public class ServerMessages {
	
	private ArrayList<ServerMessage>	messages;
	private int							nextId;
	private int							nextIndex;
	private String						file;
	
	public ServerMessages() {
		this.file = null;
		this.nextId = 1;
		this.nextIndex = 0;
		this.messages = new ArrayList<ServerMessage>();
	}
	
	public ServerMessages(String file) {
		this();
		this.file = file;
	}
	
	public synchronized void setFile(String file) {
		this.file = file;
	}
	
	public synchronized void load() throws IOException {
		if(file == null) {
			throw new IllegalStateException("No destination file set.");
		}
		BufferedReader r = new BufferedReader(new FileReader(file));
		String line;
		while((line = r.readLine()) != null) {
			if(!line.isEmpty()) {
				addMessage(line);
			}
		}
	}
	
	public synchronized void save() throws IOException {
		if(file == null) {
			throw new IllegalStateException("No target file set.");
		}
		BufferedWriter w = new BufferedWriter(new FileWriter(file));
		for(ServerMessage message : messages) {
			w.write(message.getMessage());
			w.write('\n');
		}
		w.flush();
		w.close();
	}
	
	public synchronized ServerMessage addMessage(String message) {
		ServerMessage m = new ServerMessage(nextId++, message);
		messages.add(m);
		trySave();
		return m;
	}
	
	public synchronized int size() {
		return messages.size();
	}
	
	public synchronized ArrayList<ServerMessage> getMessages() {
		return new ArrayList<ServerMessage>(messages);
	}
	
	public synchronized void setMessage(int id, String message) {
		ListIterator<ServerMessage> it = find(id);
		if(it != null) {
			it.set(new ServerMessage(id, message));
			trySave();
		} else {
			throw new NoSuchElementException("Unknown message id.");
		}
	}
	
	public synchronized void remove(int id) {
		ListIterator<ServerMessage> it = find(id);
		if(it != null) {
			it.remove();
			trySave();
		} else {
			throw new NoSuchElementException("No server message with id " + id + " exists.");
		}
	}
	
	public synchronized ServerMessage next() {
		if(messages.isEmpty()) {
			return null;
		}
		if(nextIndex >= messages.size()) {
			nextIndex = 0;
		}
		return messages.get(nextIndex++);
	}
	
	public synchronized ServerMessage get(int id) {
		ListIterator<ServerMessage> it = find(id);
		if(it == null) {
			return null;
		} else {
			return it.previous();
		}
	}
	
	private ListIterator<ServerMessage> find(int id) {
		ListIterator<ServerMessage> it = messages.listIterator();
		while(it.hasNext()) {
			if(it.next().getId() == id) {
				return it;
			}
		}
		return null;
	}
	
	private void trySave() {
		if(file != null) {
			try {
				save();
			} catch(IOException e) {}
		}
	}
	
}
