package rcon.server;

import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Properties;

public class Configuration {
	
	private String						filename;
	private EnumMap<Variable, String>	config;
	private EnumSet<Variable>			enforced;
	
	public Configuration() throws IOException {
		config = new EnumMap<Variable, String>(Variable.class);
		enforced = EnumSet.noneOf(Variable.class);
	}
	
	public synchronized void save() throws IOException {
		Properties p = new Properties();
		for(Variable v : enforced) {
			p.setProperty(v.getName(), config.get(v));
		}
		p.store(new FileWriter(filename), "Server configuration");
	}
	
	public synchronized void load(String filename) throws IOException {
		Properties p = new Properties();
		p.load(new FileReader(filename));
		for(Variable v : Variable.values()) {
			String value = p.getProperty(v.getName());
			if(value != null) {
				config.put(v, value);
				enforced.add(v);
			}
		}
		this.filename = filename;
	}
	
	public synchronized String get(Variable var) {
		return config.get(var);
	}
	
	public synchronized void enforce(Variable var, String value) {
		config.put(var, value);
		enforced.add(var);
		try {
			save();
		} catch(IOException e) {
			e.printStackTrace();
		}
	}
	
	public synchronized boolean isEnforced(Variable var) {
		return enforced.contains(var);
	}
	
	public synchronized void unenforce(Variable var) {
		enforced.remove(var);
		try {
			save();
		} catch(IOException e) {
			e.printStackTrace();
		}
	}
	
	public synchronized void update(Variable var, String value) {
		if(isEnforced(var)) {
			throw new IllegalArgumentException("Unable to update an enforced variable.");
		}
		config.put(var, value);
	}
	
	public static enum Variable {
		miniMapSpotting,
		teamKillCountForKick,
		gamePassword,
		serverDescription,
		teamKillValueForKick,
		maxPlayers,
		bulletDamage,
		nameTag,
		teamKillValueIncrease,
		regenerateHealth,
		cam3p("3pCam"),
		teamKillKickForBan,
		vehicleSpawnDelay,
		bannerUrl,
		friendlyFire,
		vehicleSpawnAllowed,
		roundRestartPlayerCount,
		idleTimeout,
		minimap,
		serverMessage,
		onlySquadLeaderSpawn,
		allUnlocksUnlocked,
		clientSideDamageArbitration,
		killCam,
		soldierHealth,
		idleBanRounds,
		spotting3d("3dSpotting"),
		hud,
		killRotation,
		playerManDownTime,
		serverName,
		roundStartPlayerCount,
		gameModeCounter,
		playerRespawnTime,
		autoBalance,
		roundsPerMap,
		teamKillValueDecreasePerSecond;
		
		private static HashMap<String, Variable>	variables	= new HashMap<String, Variable>();
		static {
			for(Variable var : values()) {
				variables.put(var.getName(), var);
			}
		}
		
		public static Variable get(String name) {
			return variables.get(name);
		}
		
		private final String	realName;
		
		private Variable() {
			realName = null;
		}
		
		private Variable(String realName) {
			this.realName = realName;
		}
		
		public String getName() {
			return realName == null ? name() : realName;
		}
		
	}
}
