package rcon.server;

import java.io.IOException;
import java.net.Socket;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.log4j.Logger;

import rcon.Hex;
import rcon.commands.ClientRequest;
import rcon.commands.Command;
import rcon.net.RCONInputStream;
import rcon.net.RCONOutputStream;

public class ServerConnection implements Runnable {
	
	static Logger							logger	= Logger.getLogger(ServerConnection.class);
	
	private Socket							socket;
	private LinkedBlockingQueue<Command>	events;
	private SynchronousQueue<Command>		responses;
	private AtomicInteger					seq;
	private ReceiverThread					receiver;
	private RCONOutputStream				rout;
	private SimpleTimer						timeoutTimer;
	
	public ServerConnection(int timeout) {
		events = new LinkedBlockingQueue<Command>();
		responses = new SynchronousQueue<Command>();
		seq = new AtomicInteger(1);
		this.timeoutTimer = new SimpleTimer(timeout, this);
	}
	
	public boolean awaitShutdown() {
		try {
			receiver.join();
			return true;
		} catch(InterruptedException e) {
			return false;
		}
	}
	
	protected void addResponse(Command response) {
		timeoutTimer.abort();
		responses.offer(response);
	}
	
	public Command sendRequest(Command command) throws IOException {
		long before = System.currentTimeMillis();
		command.setSeq(seq.getAndIncrement());
		try {
			rout.sendCommand(command);
			timeoutTimer.start();
			Command response = responses.take();
			if(response == Command.EOF) {
				throw new IOException("End of file.");
			} else {
				return response;
			}
		} catch(InterruptedException e) {
			throw new IOException("Interrupted while sending request.", e);
		} catch(IOException e) {
			throw new IOException(e);
		} finally {
			logger.trace("It took " + (System.currentTimeMillis() - before) + " ms to send receive a response. ("
					+ command.getName() + ")");
		}
	}
	
	protected void sentResponse(Command response) throws IOException {
		rout.sendCommand(response);
	}
	
	protected void addEvent(Command event) {
		events.add(event);
	}
	
	public Command getEvent() {
		try {
			return events.take();
		} catch(InterruptedException e) {
			return Command.EOF;
		}
	}
	
	public void start(String ip, int port) throws IOException {
		socket = new Socket(ip, port);
		rout = new RCONOutputStream(socket.getOutputStream());
		RCONInputStream rin = new RCONInputStream(socket.getInputStream());
		
		receiver = new ReceiverThread(rin, this);
		receiver.start();
	}
	
	public boolean login(String password) throws IOException {
		Command result = sendRequest(new ClientRequest("login.hashed"));
		if(result.is("OK")) {
			String salt = result.get(0);
			String hexHash = Hex.hashPassword(password, salt);
			result = sendRequest(new ClientRequest("login.hashed", hexHash));
			if(result.is("OK")) {
				return true;
			}
		}
		return false;
	}
	
	public boolean logout() throws IOException {
		Command result = sendRequest(new ClientRequest("logout"));
		return result.is("OK");
	}
	
	public void quit() throws IOException {
		Command result = sendRequest(new ClientRequest("quit"));
		if(!result.is("OK")) {
			System.out.println("Server did not acknowledge quit command. Forcing shutdown.");
		}
	}
	
	public void stop() {
		receiver.interrupt();
		
		try {
			socket.close();
		} catch(IOException e) {
			logger.error("Failed to close socket.", e);
		}
		
		try {
			receiver.join(500);
		} catch(InterruptedException e) {
			logger.error("Failed to shut down network threads.", e);
			throw new RuntimeException(e);
		}
		timeoutTimer.close();
	}
	
	@Override
	public void run() {
		responses.offer(Command.EOF);
	}
	
}
