package rcon.client;

import java.io.IOException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.Iterator;

import rcon.RCONDaemon;
import rcon.commands.Command;
import rcon.net.RCONOutputStream;
import rcon.players.Admin;
import rcon.players.Right;

public class AdminModule extends CommandModule {
	
	public AdminModule(RCONOutputStream out, RCONDaemon daemon) {
		super(out, daemon);
	}
	
	public void edit(Admin user, Command request) throws IOException {
		if(request.size() == 2) {
			try {
				int id = request.getInt(0);
				String guid = request.get(1);
				Admin admin = daemon.getDatabase().admins.get(id);
				if(admin == null) {
					reply(request, "AdminNotFound");
				} else {
					admin.setGuid(guid);
					if(daemon.getDatabase().admins.save(admin)) {
						log(user, "Edited GUID of " + admin.getUserName() + " to '" + admin.getGuid() + "'.");
						reply(request, "OK");
					} else {
						reply(request, "ServerError");
					}
				}
			} catch(NumberFormatException e) {
				reply(request, "InvalidArguments");
			} catch(SQLException e) {
				reply(request, "ServerError");
			}
		} else {
			reply(request, "InvalidArguments");
		}
	}
	
	public void register(Admin user, Command request) throws IOException {
		if(request.size() == 3) {
			String guid = request.get(0);
			String username = request.get(1);
			String password = request.get(2);
			try {
				Admin adminByGuid = daemon.getDatabase().admins.getByGuid(guid);
				Admin adminByName = daemon.getDatabase().admins.get(username);
				if(adminByGuid != null) {
					reply(request, "GuidAlreadyExists");
				} else if(adminByName != null) {
					reply(request, "NameAlreadyExists");
				} else {
					Admin admin = new Admin(0, guid, username, password, EnumSet.noneOf(Right.class));
					if(daemon.getDatabase().admins.create(admin)) {
						log(user, "Registered admin '" + username + "' with GUID '" + guid + "'.");
						reply(request, "OK", String.valueOf(admin.getId()));
					} else {
						reply(request, "ServerError");
					}
				}
			} catch(SQLException e) {
				reply(request, "ServerError");
			}
		} else {
			reply(request, "InvalidArguments");
		}
	}
	
	public void setPassword(Admin user, Command request) throws IOException {
		if(request.size() == 2) {
			try {
				int id = request.getInt(0);
				String password = request.get(1);
				Admin admin = daemon.getDatabase().admins.get(id);
				if(admin == null) {
					reply(request, "AdminNotFound");
				} else if(user != null && (user.getId() == admin.getId() || user.hasRight(Right.editAdmin))) {
					admin.setPassword(password);
					if(daemon.getDatabase().admins.save(admin)) {
						log(user, "Changed password for '" + admin.getUserName() + "'.");
						reply(request, "OK");
					} else {
						reply(request, "ServerError");
					}
				} else {
					reply(request, "NotAuthorized");
				}
			} catch(NumberFormatException e) {
				reply(request, "InvalidArguments");
			} catch(SQLException e) {
				reply(request, "ServerError");
			}
		} else {
			reply(request, "InvalidArguments");
		}
	}
	
	public void get(Admin user, Command request) throws IOException {
		if(request.size() == 2) {
			String method = request.get(0);
			try {
				Admin admin = null;
				if(method.equals("name")) {
					String name = request.get(1);
					admin = daemon.getDatabase().admins.get(name);
				} else if(method.equals("id")) {
					int id = request.getInt(1);
					admin = daemon.getDatabase().admins.get(id);
				} else if(method.equals("eaguid")) {
					String guid = request.get(1);
					admin = daemon.getDatabase().admins.getByGuid(guid);
				} else {
					reply(request, "InvalidMethod");
				}
				if(admin == null) {
					reply(request, "AdminNotFound");
				} else {
					EnumSet<Right> rights = admin.getRights();
					
					String[] args = new String[4 + rights.size()];
					
					args[0] = String.valueOf(admin.getId());
					args[1] = admin.getGuid();
					args[2] = admin.getUserName();
					args[3] = String.valueOf(rights.size());
					
					Iterator<Right> it = rights.iterator();
					for(int i = 4; i < args.length; i++) {
						args[i] = it.next().name();
					}
					reply(request, "OK", args);
				}
			} catch(NumberFormatException e) {
				reply(request, "InvalidArguments");
			} catch(SQLException e) {
				reply(request, "ServerError");
			}
		} else {
			reply(request, "InvalidArguments");
		}
	}
	
	public void setRights(Admin user, Command request) throws IOException {
		if(request.size() >= 2) {
			try {
				int adminId = request.getInt(0);
				int rightsCount = request.getInt(1);
				if(request.size() == 2 + rightsCount) {
					String[] rightNames = new String[rightsCount];
					System.arraycopy(request.getArguments(), 2, rightNames, 0, rightsCount);
					EnumSet<Right> rights = Right.parseRights(rightNames);
					Admin admin = daemon.getDatabase().admins.get(adminId);
					if(admin == null) {
						reply(request, "AdminNotFound");
					} else {
						admin.setRights(rights);
						if(daemon.getDatabase().admins.save(admin)) {
							log(user, "Set the rights of '" + admin.getUserName() + "' to '" + Right.toString(rights)
									+ "'.");
							reply(request, "OK", String.valueOf(admin.getRights().size()));
						} else {
							reply(request, "ServerError");
						}
					}
				} else {
					reply(request, "InvalidArguments");
				}
			} catch(NumberFormatException e) {
				reply(request, "InvalidArguments");
			} catch(SQLException e) {
				reply(request, "ServerError");
			}
		} else {
			reply(request, "InvalidArguments");
		}
	}
	
	public void list(Admin user, Command request) throws IOException {
		try {
			ArrayList<Admin> admins = daemon.getDatabase().admins.getAll();
			String[] args = new String[1 + admins.size() * 3];
			int i = 0;
			args[i++] = String.valueOf(admins.size());
			for(Admin a : admins) {
				args[i++] = String.valueOf(a.getId());
				args[i++] = a.getGuid();
				args[i++] = a.getUserName();
			}
			reply(request, "OK", args);
		} catch(SQLException e) {
			reply(request, "ServerError");
		}
	}
	
}
