package rcon.client;

import java.io.IOException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.Iterator;

import rcon.RCONDaemon;
import rcon.commands.Command;
import rcon.event.BanDuration;
import rcon.net.RCONOutputStream;
import rcon.players.Admin;
import rcon.players.Player;
import rcon.players.Punishment;
import rcon.players.PunishmentType;

public class BanModule extends CommandModule {
	
	public BanModule(RCONOutputStream out, RCONDaemon daemon) {
		super(out, daemon);
	}
	
	public void history(Admin user, Command request) throws IOException {
		if(request.size() == 1) {
			String name = request.get(0);
			Player p = daemon.getServer().getOnline().getByName(name);
			try {
				ArrayList<Punishment> punishments;
				if(p != null) {
					punishments = daemon.getDatabase().bans.getByPlayer(p);
				} else {
					punishments = daemon.getDatabase().bans.getByName(name, false);
				}
				int warns = 0;
				int kicks = 0;
				int bans = 0;
				for(Punishment punishment : punishments) {
					if(punishment.isEnabled()) {
						switch(punishment.getType()) {
							case warn:
								warns++;
								break;
							case kick:
								kicks++;
								break;
							case ban:
								bans++;
								break;
						}
					}
				}
				reply(request, "OK", String.valueOf(warns), String.valueOf(kicks), String.valueOf(bans));
			} catch(SQLException e) {
				reply(request, "ServerError");
			}
		} else {
			reply(request, "InvalidArguments");
		}
	}
	
	public void remove(Admin user, Command request) throws IOException {
		if(request.size() == 1) {
			try {
				int banId = request.getInt(0);
				if(daemon.getDatabase().bans.remove(banId)) {
					log(user, "Removed warn/kick/ban with id " + banId + ".");
					reply(request, "OK");
				} else {
					reply(request, "InvalidBan");
				}
			} catch(NumberFormatException e) {
				reply(request, "InvalidBan");
			} catch(SQLException e) {
				reply(request, "ServerError");
			}
		} else {
			reply(request, "InvalidArguments");
		}
	}
	
	public void list(Admin user, Command request) throws IOException {
		if(request.size() == 3) {
			String method = request.get(0);
			String searchString = request.get(1);
			String type = request.get(2);
			EnumSet<PunishmentType> types = EnumSet.noneOf(PunishmentType.class);
			if(type.equals("all")) {
				types = EnumSet.allOf(PunishmentType.class);
			} else {
				try {
					types = EnumSet.of(Enum.valueOf(PunishmentType.class, type.toLowerCase()));
				} catch(Exception e) {
					reply(request, "InvalidArguments");
				}
			}
			if(!types.isEmpty()) {
				try {
					ArrayList<Punishment> punishments;
					if(method.equals("exact")) {
						punishments = daemon.getDatabase().bans.getByName(searchString, false);
					} else if(method.equals("partial")) {
						punishments = daemon.getDatabase().bans.getByName(searchString, true);
					} else if(method.equals("eaguid")) {
						punishments = daemon.getDatabase().bans.getByEaGuid(searchString);
					} else if(method.equals("pbguid")) {
						punishments = daemon.getDatabase().bans.getByPbGuid(searchString);
					} else if(method.equals("last")) {
						int amount = Math.min(request.getInt(1), 50);
						punishments = daemon.getDatabase().bans.getLast(types, amount);
					} else if(method.equals("issuer")) {
						punishments = daemon.getDatabase().bans.getByAdmin(request.getInt(1));
					} else {
						reply(request, "InvalidMethod");
						return;
					}
					Iterator<Punishment> it = punishments.iterator();
					while(it.hasNext()) {
						if(!types.contains(it.next().getType())) {
							it.remove();
						}
					}
					ArrayList<String> arguments = new ArrayList<String>();
					arguments.add(String.valueOf(punishments.size()));
					for(Punishment p : punishments) {
						arguments.add(String.valueOf(p.getId()));
						arguments.add(p.getName());
						arguments.add(p.getEaGuid());
						arguments.add(p.getPbGuid());
						arguments.add(p.getType().name());
						arguments.add(String.valueOf(p.getStart()));
						arguments.add(String.valueOf(p.getEnd()));
						arguments.add(p.getReason());
						arguments.add(String.valueOf(p.getIssuer().getId()));
						arguments.add(p.getIssuer().getUserName());
						arguments.add(String.valueOf(p.isEnabled()));
					}
					reply(request, "OK", arguments.toArray(new String[arguments.size()]));
				} catch(SQLException e) {
					reply(request, "ServerError");
				} catch(NumberFormatException e) {
					reply(request, "InvalidArguments");
				}
			}
		} else {
			reply(request, "InvalidArguments");
		}
	}
	
	public void ban(Admin user, Command request) throws IOException {
		if(request.size() == 5) {
			String name = request.get(0);
			String eaGuid = request.get(1);
			String pbGuid = request.get(2);
			String duration = request.get(3);
			String reason = request.get(4);
			int time;
			boolean permanent = duration.equals("perm");
			if(permanent) {
				time = 0;
			} else {
				time = Integer.parseInt(duration);
			}
			
			long start = System.currentTimeMillis() / 1000;
			long end = permanent ? 0 : (start + time);
			
			Punishment ban = new Punishment(name, eaGuid, pbGuid, PunishmentType.ban, start, end, reason, user, true);
			try {
				if(daemon.getDatabase().bans.add(ban)) {
					Player p = daemon.getServer().getOnline().getByName(name);
					if(p != null) {
						String timeString = timeToDuration(time);
						daemon.getServer().ban(p, reason, timeString);
					}
					log(user, "Banned '" + name + "' (" + eaGuid + ", " + pbGuid + ") for " + reason + ". Duration: "
							+ duration + ".");
					reply(request, "OK", String.valueOf(ban.getId()));
				} else {
					reply(request, "ServerError");
				}
			} catch(SQLException e) {
				reply(request, "ServerError");
			}
			
		} else {
			reply(request, "InvalidArguments");
		}
	}
	
	private String timeToDuration(int time) {
		int amount = 0;
		String name = "";
		if(time >= BanDuration.month.getDuration()) {
			amount = time / BanDuration.month.getDuration();
			name = "month";
		} else if(time >= BanDuration.week.getDuration()) {
			amount = time / BanDuration.week.getDuration();
			name = "week";
		} else if(time >= BanDuration.day.getDuration()) {
			amount = time / BanDuration.day.getDuration();
			name = "day";
		} else if(time >= BanDuration.hour.getDuration()) {
			amount = time / BanDuration.hour.getDuration();
			name = "hour";
		} else if(time >= BanDuration.moment.getDuration()) {
			amount = 1;
			name = "moment";
		} else {
			name = "perm";
		}
		if(name.equals("perm")) {
			return name;
		} else {
			return (amount > 1 ? String.valueOf(amount) : "a") + " " + name + (amount > 1 ? "s" : "");
		}
	}
	
	public void kick(Admin user, Command request) throws IOException {
		if(request.size() == 2) {
			String name = request.get(0);
			String reason = request.get(1);
			
			Player p = daemon.getServer().getOnline().getByName(name);
			if(p == null) {
				reply(request, "PlayerNotFound");
			} else {
				Punishment kick = Punishment.kick(p, reason, user);
				try {
					if(daemon.getDatabase().bans.add(kick)) {
						daemon.getServer().kick(p, reason);
						log(user, "Kicked '" + name + "' (" + p.getEaGuid() + ", " + p.getPbGuid() + ") for " + reason
								+ ".");
						reply(request, "OK", String.valueOf(kick.getId()));
					} else {
						reply(request, "ServerError");
					}
				} catch(SQLException e) {
					reply(request, "ServerError");
				}
			}
		} else {
			reply(request, "InvalidArguments");
		}
	}
	
	public void warn(Admin user, Command request) throws IOException {
		if(request.size() == 2) {
			String name = request.get(0);
			String reason = request.get(1);
			
			Player p = daemon.getServer().getOnline().getByName(name);
			if(p == null) {
				reply(request, "PlayerNotFound");
			} else {
				Punishment warn = Punishment.warn(p, reason, user);
				try {
					if(daemon.getDatabase().bans.add(warn)) {
						daemon.getServer().warn(p, reason);
						log(user, "Warned '" + name + "' (" + p.getEaGuid() + ", " + p.getPbGuid() + ") for " + reason
								+ ".");
						reply(request, "OK", String.valueOf(warn.getId()));
					} else {
						reply(request, "ServerError");
					}
				} catch(SQLException e) {
					reply(request, "ServerError");
				}
			}
		} else {
			reply(request, "InvalidArguments");
		}
	}
	
	public void edit(Admin user, Command request) throws IOException {
		if(request.size() == 7) {
			try {
				int id = request.getInt(0);
				String name = request.get(1);
				String eaGuid = request.get(2);
				String pbGuid = request.get(3);
				int end = request.getInt(4);
				String reason = request.get(5);
				String enabled = request.get(6);
				Punishment p = daemon.getDatabase().bans.get(id);
				if(p == null) {
					reply(request, "BanNotFound");
				} else {
					p.setName(name);
					p.setEaGuid(eaGuid);
					p.setPbGuid(pbGuid);
					p.setEnd(end);
					p.setReason(reason);
					p.setEnabled(enabled.equalsIgnoreCase("true"));
					if(daemon.getDatabase().bans.save(p)) {
						log(user, "Edited ban id " + id + ". Applies to player '" + name + "' (" + eaGuid + ", "
								+ pbGuid + ")");
						reply(request, "OK");
					} else {
						reply(request, "ServerError");
					}
				}
			} catch(SQLException e) {
				reply(request, "ServerError");
			} catch(NumberFormatException e) {
				reply(request, "InvalidArguments");
			}
		} else {
			reply(request, "InvalidArguments");
		}
	}
	
	public void get(Admin user, Command request) throws IOException {
		if(request.size() == 1) {
			try {
				int id = request.getInt(0);
				Punishment p = daemon.getDatabase().bans.get(id);
				if(p == null) {
					reply(request, "BanNotFound");
				} else {
					String[] args = new String[] {p.getName(), p.getEaGuid(), p.getPbGuid(), p.getType().name(),
							String.valueOf(p.getStart()), String.valueOf(p.getEnd()), p.getReason(),
							String.valueOf(p.getIssuer().getId()), p.getIssuer().getUserName(),
							String.valueOf(p.isEnabled())};
					reply(request, "OK", args);
				}
			} catch(NumberFormatException e) {
				reply(request, "InvalidArguments");
			} catch(SQLException e) {
				reply(request, "ServerError");
			}
		} else {
			reply(request, "InvalidArguments");
		}
	}
	
}
