package rcon.map;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

public class Maps {
	
	private String		filename;
	private Playlist	playlist;
	
	public Maps(String filename) {
		this.filename = filename;
		this.playlist = new Playlist();
	}
	
	public Playlist getPlaylist() {
		return playlist;
	}
	
	public void load() throws IOException {
		
		Document doc;
		try {
			doc = getDocument(filename);
		} catch(ParserConfigurationException e) {
			throw new IOException(e);
		} catch(SAXException e) {
			throw new IOException(e);
		}
		
		synchronized(playlist) {
			playlist.clear();
			Element root = doc.getDocumentElement();
			if(root.getNodeName().equals("maps")) {
				
				Node node = root.getFirstChild();
				while(node != null) {
					if(node.getNodeType() == Node.ELEMENT_NODE && node.getNodeName().equals("map")) {
						Element mapNode = (Element)node;
						parseMap(mapNode);
					}
					node = node.getNextSibling();
				}
			} else {
				throw new IOException("Invalid root node.");
			}
		}
	}
	
	private Document getDocument(String filename) throws ParserConfigurationException, SAXException, IOException {
		DocumentBuilder builder = getDocumentBuilder();
		Document doc = builder.parse(filename);
		return doc;
	}
	
	private DocumentBuilder getDocumentBuilder() throws ParserConfigurationException {
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = dbf.newDocumentBuilder();
		return builder;
	}
	
	private void parseMap(Element mapNode) {
		String name = mapNode.getAttribute("name");
		String roundString = mapNode.getAttribute("rounds");
		int rounds = 2;
		if(!roundString.isEmpty()) {
			try {
				rounds = Integer.parseInt(roundString);
			} catch(NumberFormatException e) {
				e.printStackTrace();
			}
		}
		Map map;
		if(name.isEmpty()) {
			map = null;
		} else {
			map = Map.get(name);
		}
		if(map == null) {
			System.out.println("Unknown or invalid map '" + name + "'. Skipping.");
		} else {
			playlist.append(map, rounds);
		}
	}
	
	public void save() throws IOException {
		Document doc;
		try {
			doc = getDocumentBuilder().newDocument();
		} catch(ParserConfigurationException e) {
			throw new IOException(e);
		}
		
		synchronized(playlist) {
			Element root = doc.createElement("maps");
			for(Playlist.Entry map : playlist.get()) {
				Element mapNode = doc.createElement("map");
				mapNode.setAttribute("name", map.getMap().getLevel());
				mapNode.setAttribute("rounds", String.valueOf(map.getRounds()));
				root.appendChild(mapNode);
			}
			
		}
		saveDocument(doc, filename);
	}
	
	private void saveDocument(Document doc, String filename) throws IOException {
		try {
			TransformerFactory tf = TransformerFactory.newInstance();
			Transformer t = tf.newTransformer();
			DOMSource source = new DOMSource(doc);
			FileWriter fw = new FileWriter(filename);
			BufferedWriter w = new BufferedWriter(fw);
			StreamResult result = new StreamResult(w);
			t.transform(source, result);
			w.close();
			fw.close();
		} catch(TransformerConfigurationException e) {
			throw new IOException(e);
		} catch(TransformerException e) {
			throw new IOException(e);
		}
	}
	
}
