package rcon.players;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.EnumSet;

import rcon.database.Database;
import rcon.database.DatabaseModule;

public class PunishmentDatabase extends DatabaseModule {
	
	public PunishmentDatabase(Database database, String id) {
		super(database, id);
	}
	
	public ArrayList<Punishment> getActive(Player player, PunishmentType type) throws SQLException {
		String query = "SELECT * FROM bans, admins WHERE ((bans.eaguid = ? AND CHAR_LENGTH(bans.eaguid) > 0) "
				+ "OR (bans.pbguid = ? AND CHAR_LENGTH(bans.pbguid) > 0) "
				+ "OR (bans.name = ? AND CHAR_LENGTH(bans.name) > 0) ) AND bans.type = ? AND "
				+ "admins.id = bans.issuer AND (bans.endtime > ? OR bans.endtime = 0) AND bans.enabled = TRUE";
		Connection con = getConnection();
		PreparedStatement stmt = null;
		ResultSet result = null;
		try {
			stmt = con.prepareStatement(query);
			stmt.setString(1, player.getEaGuid());
			stmt.setString(2, player.getPbGuid());
			stmt.setString(3, player.getName());
			stmt.setString(4, type.name());
			stmt.setLong(5, System.currentTimeMillis() / 1000);
			
			result = stmt.executeQuery();
			return parsePunishments(result);
			
		} finally {
			close(con, stmt, result);
		}
	}
	
	public ArrayList<Punishment> getByPlayer(Player player) throws SQLException {
		String query = "SELECT * FROM bans, admins WHERE "
				+ "(bans.eaguid = ? OR bans.pbguid = ? OR bans.name = ?) AND admins.id = bans.issuer";
		Connection con = getConnection();
		PreparedStatement stmt = null;
		ResultSet result = null;
		try {
			stmt = con.prepareStatement(query);
			stmt.setString(1, player.getEaGuid());
			stmt.setString(2, player.getPbGuid());
			stmt.setString(3, player.getName());
			
			result = stmt.executeQuery();
			return parsePunishments(result);
			
		} finally {
			close(con, stmt, result);
		}
	}
	
	public ArrayList<Punishment> getByName(String name, boolean partial) throws SQLException {
		String query;
		if(partial) {
			// query =
			// "SELECT * FROM bans, admins WHERE admins.id = bans.issuer AND NOT POSITION(LCASE(?) IN LCASE(bans.name)) = 0";
			query = "SELECT * FROM bans, admins WHERE admins.id = bans.issuer AND bans.name LIKE ?";
		} else {
			query = "SELECT * FROM bans, admins WHERE bans.name = ? AND admins.id = bans.issuer";
		}
		Connection con = getConnection();
		PreparedStatement stmt = null;
		ResultSet result = null;
		try {
			stmt = con.prepareStatement(query);
			if(partial) {
				stmt.setString(1, name + "%");
			} else {
				stmt.setString(1, name);
			}
			
			result = stmt.executeQuery();
			return parsePunishments(result);
			
		} finally {
			close(con, stmt, result);
		}
	}
	
	public ArrayList<Punishment> getByPbGuid(String guid) throws SQLException {
		String query = "SELECT * FROM bans, admins WHERE bans.pbguid = ? AND admins.id = bans.issuer";
		Connection con = getConnection();
		PreparedStatement stmt = null;
		ResultSet result = null;
		try {
			stmt = con.prepareStatement(query);
			stmt.setString(1, guid);
			
			result = stmt.executeQuery();
			return parsePunishments(result);
			
		} finally {
			close(con, stmt, result);
		}
	}
	
	public ArrayList<Punishment> getByEaGuid(String guid) throws SQLException {
		String query = "SELECT * FROM bans, admins WHERE bans.eaguid = ? AND admins.id = bans.issuer";
		Connection con = getConnection();
		PreparedStatement stmt = null;
		ResultSet result = null;
		try {
			stmt = con.prepareStatement(query);
			stmt.setString(1, guid);
			
			result = stmt.executeQuery();
			return parsePunishments(result);
			
		} finally {
			close(con, stmt, result);
		}
	}
	
	public ArrayList<Punishment> getByAdmin(int adminId) throws SQLException {
		Connection con = getConnection();
		PreparedStatement stmt = null;
		ResultSet result = null;
		try {
			stmt = con.prepareStatement("SELECT * FROM bans, admins WHERE bans.issuer = ? AND admins.id = bans.issuer");
			stmt.setInt(1, adminId);
			
			result = stmt.executeQuery();
			return parsePunishments(result);
			
		} finally {
			close(con, stmt, result);
		}
	}
	
	public ArrayList<Punishment> getLast(EnumSet<PunishmentType> types, int amount) throws SQLException {
		Connection con = getConnection();
		PreparedStatement stmt = null;
		ResultSet result = null;
		try {
			StringBuilder sb = new StringBuilder();
			sb.append("SELECT * FROM bans, admins WHERE bans.type IN (");
			for(int i = 0; i < types.size(); i++) {
				if(i != 0) {
					sb.append(", ");
				}
				sb.append("?");
			}
			sb.append(") AND admins.id = bans.issuer LIMIT ?");
			stmt = con.prepareStatement(sb.toString());
			
			int i = 1;
			for(PunishmentType t : types) {
				stmt.setString(i++, t.name());
			}
			stmt.setInt(i++, amount);
			
			result = stmt.executeQuery();
			return parsePunishments(result);
			
		} finally {
			close(con, stmt, result);
		}
	}
	
	public Punishment get(int id) throws SQLException {
		Connection con = getConnection();
		PreparedStatement stmt = null;
		ResultSet result = null;
		try {
			stmt = con.prepareStatement("SELECT * FROM bans, admins WHERE bans.id = ? AND admins.id = bans.issuer");
			stmt.setInt(1, id);
			
			result = stmt.executeQuery();
			if(result.next()) {
				return parse(result);
			} else {
				return null;
			}
		} finally {
			close(con, stmt, result);
		}
	}
	
	public boolean add(Punishment punishment) throws SQLException {
		String query = "INSERT INTO bans "
				+ "(name, eaguid, pbguid, type, starttime, endtime, reason, issuer, enabled) "
				+ "VALUES(?, ?, ?, ?, ?, ?, ?, ?, ?)";
		Connection con = getConnection();
		PreparedStatement stmt = null;
		ResultSet keys = null;
		try {
			stmt = con.prepareStatement(query, PreparedStatement.RETURN_GENERATED_KEYS);
			stmt.setString(1, punishment.getName());
			stmt.setString(2, punishment.getEaGuid());
			stmt.setString(3, punishment.getPbGuid());
			stmt.setString(4, punishment.getType().name());
			stmt.setLong(5, punishment.getStart());
			stmt.setLong(6, punishment.getEnd());
			stmt.setString(7, punishment.getReason());
			stmt.setInt(8, punishment.getIssuer().getId());
			stmt.setBoolean(9, punishment.isEnabled());
			stmt.executeUpdate();
			
			keys = stmt.getGeneratedKeys();
			if(keys.next()) {
				punishment.setId(keys.getInt(1));
				return true;
			} else {
				return false;
			}
		} finally {
			close(con, stmt, keys);
		}
	}
	
	public boolean save(Punishment p) throws SQLException {
		String query = "UPDATE bans SET name = ?, eaguid = ?, pbguid = ?, type = ?, starttime = ?, "
				+ "endtime = ?, reason = ?, issuer = ?, enabled = ? WHERE id = ?";
		Connection con = getConnection();
		PreparedStatement stmt = null;
		ResultSet result = null;
		try {
			stmt = con.prepareStatement(query);
			stmt.setString(1, p.getName());
			stmt.setString(2, p.getEaGuid());
			stmt.setString(3, p.getPbGuid());
			stmt.setString(4, p.getType().name());
			stmt.setLong(5, p.getStart());
			stmt.setLong(6, p.getEnd());
			stmt.setString(7, p.getReason());
			stmt.setInt(8, p.getIssuer().getId());
			stmt.setBoolean(9, p.isEnabled());
			stmt.setInt(10, p.getId());
			return stmt.executeUpdate() == 1;
			
		} finally {
			close(con, stmt, result);
		}
	}
	
	public boolean remove(int id) throws SQLException {
		Connection con = getConnection();
		PreparedStatement stmt = null;
		ResultSet result = null;
		try {
			stmt = con.prepareStatement("UPDATE bans SET enabled = FALSE WHERE id = ?");
			stmt.setInt(1, id);
			return stmt.executeUpdate() == 1;
		} finally {
			close(con, stmt, result);
		}
	}
	
	public String getReason(String shortcut) throws SQLException {
		Connection con = getConnection();
		PreparedStatement stmt = null;
		ResultSet result = null;
		try {
			stmt = con.prepareStatement("SELECT * FROM reasons WHERE shortcut = ?");
			stmt.setString(1, shortcut.toLowerCase());
			result = stmt.executeQuery();
			if(result.next()) {
				return result.getString("reasons.explaination");
			} else {
				return shortcut;
			}
		} finally {
			close(con, stmt, result);
		}
	}
	
	private ArrayList<Punishment> parsePunishments(ResultSet result) throws SQLException {
		ArrayList<Punishment> punishments = new ArrayList<Punishment>();
		if(result.next()) {
			do {
				Punishment punishment = parse(result);
				punishments.add(punishment);
			} while(result.next());
		}
		return punishments;
	}
	
	private Punishment parse(ResultSet result) throws SQLException {
		Admin admin = database.admins.parse(result);
		int id = result.getInt("bans.id");
		String bannedName = result.getString("bans.name");
		String bannedEaGuid = result.getString("bans.eaguid");
		String bannedPbGuid = result.getString("bans.pbguid");
		String typeString = result.getString("bans.type");
		PunishmentType type = Enum.valueOf(PunishmentType.class, typeString);
		long start = result.getLong("bans.starttime");
		long end = result.getLong("bans.endtime");
		String reason = result.getString("bans.reason");
		boolean enabled = result.getBoolean("bans.enabled");
		Punishment punishment = new Punishment(bannedName, bannedEaGuid, bannedPbGuid, type, start, end, reason, admin,
			enabled);
		punishment.setId(id);
		return punishment;
	}
	
}
