package rcon.server;

import java.util.ArrayList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;

import rcon.commands.Command;
import rcon.event.RCONEventHandler;
import rcon.players.Player;
import rcon.players.PlayerInfo;

public class EventParser extends Thread {
	
	static Logger								logger			= Logger.getLogger(EventParser.class);
	
	public static final String					pListRegex		= "^.*PunkBuster Server:\\s+(\\d+)\\s+([0-9a-f]+)\\(-\\)\\s+([0-9\\.]+):(\\d+)\\s+[^\"]+\"([^\"]+)\"\\s*$";
	private static final Pattern				pListPattern	= Pattern.compile(pListRegex);
	
	public static final String					guidRegex		= "^.*PunkBuster Server: Player GUID Computed\\s+([0-9a-f]+)\\(-\\)\\s+\\(slot #(\\d+)\\)\\s+([0-9\\.]+):(\\d+)\\s+(.+)\\s*$";
	private static final Pattern				guidPattern		= Pattern.compile(guidRegex);
	
	private final ArrayList<RCONEventHandler>	handlers;
	private final ServerConnection				connection;
	
	private final OnlineList					online;
	
	public EventParser(OnlineList online, ServerConnection connection) {
		super("EventParser");
		handlers = new ArrayList<RCONEventHandler>();
		this.online = online;
		this.connection = connection;
	}
	
	@Override
	public void run() {
		while(true) {
			try {
				Command event = connection.getEvent();
				if(event == Command.EOF) {
					break;
				}
				
				/*
				 * try { connection.sentResponse(new Command("OK",
				 * MessageType.client_response, event.getSeq())); }
				 * catch(IOException e) { e.printStackTrace(); }
				 */
				
				String cmd = event.getName();
				String[] args = event.getArguments();
				
				if(cmd.equals("player.onJoin")) {
					onJoin(args[0], args[1]);
				} else if(cmd.equals("player.onLeave")) {
					PlayerInfo info = PlayerInfo.parseSingle(event);
					onLeave(args[0], info);
				} else if(cmd.equals("player.onKill")) {
					String killer = args[0];
					String killed = args[1];
					String weapon = args[2];
					String headshot = args[3];
					boolean hs = Boolean.valueOf(headshot);
					onKill(killer, killed, weapon, hs);
				} else if(cmd.equals("player.onChat")) {
					onChat(args[0], args[1]);
				} else if(cmd.equals("punkBuster.onMessage")) {
					onPunkbuster(args[0]);
				} else if(cmd.equals("player.onAuthenticated")) {
					onAuthenticated(args[0]);
				} else if(cmd.equals("player.onTeamChange")) {
					onTeamChange(args[0], args[1], args[2]);
				} else if(cmd.equals("player.onSquadChange")) {
					onSquadChange(args[0], args[1], args[2]);
				} else if(cmd.equals("player.onKicked")) {
					onKicked(args[0], args[1]);
				} else if(cmd.equals("server.onLevelLoaded")) {
					String levelName = args[0];
					String gameMode = args[1];
					int roundsPlayed = Integer.parseInt(args[1]);
					int roundsTotal = Integer.parseInt(args[2]);
					onLevelLoaded(levelName, gameMode, roundsPlayed, roundsTotal);
				} else if(cmd.equals("player.onSpawn")) {
					String name = args[0];
					onSpawn(name);
				} else if(cmd.equals("server.onRoundOver")) {
					String winner = args[0];
					// onRoundOver(winner);
				} else if(cmd.equals("server.onRoundOverPlayers")) {
					ArrayList<PlayerInfo> info = PlayerInfo.parse(event);
					onRoundOverPlayers(info);
				} else if(cmd.equals("server.onRoundOverTeamScores")) {
					// onRoundOverTeamScores(teamScores);
				} else {
					logger.warn("Unknown event notification: " + cmd);
				}
			} catch(Exception e) {
				logger.error("Fatal error during event parsing. Ignoring.", e);
			}
		}
		logger.trace("Event parsing stopped.");
	}
	
	private void onRoundOverPlayers(ArrayList<PlayerInfo> info) {
		for(PlayerInfo i : info) {
			onPlayerInfo(i);
		}
		for(RCONEventHandler handler : handlers) {
			handler.onRoundOverPlayers(info);
		}
	}
	
	private void onSpawn(String name) {
		Player p = online.getByName(name);
		if(p != null) {
			boolean first = !p.hasSpawned();
			for(RCONEventHandler handler : handlers) {
				handler.onSpawn(p, first);
			}
			p.spawn();
		}
	}
	
	private void onLevelStarted() {
		for(RCONEventHandler handler : handlers) {
			handler.onLevelStarted();
		}
	}
	
	private void onLevelLoaded(String level, String gameMode, int roundsPlayed, int roundsTotal) {
		for(RCONEventHandler handler : handlers) {
			handler.onLevelLoaded(level, gameMode, roundsPlayed, roundsTotal);
		}
	}
	
	private void onSquadChange(String name, String teamId, String squadId) {
		Player p = online.getByName(name);
		if(p != null) {
			p.onUpdated();
			int team = Integer.parseInt(teamId);
			int squad = Integer.parseInt(squadId);
			for(RCONEventHandler handler : handlers) {
				handler.onSquadChange(p, team, squad);
			}
		}
	}
	
	private void onTeamChange(String name, String teamId, String squadId) {
		Player p = online.getByName(name);
		if(p != null) {
			p.onUpdated();
			int team = Integer.parseInt(teamId);
			int squad = Integer.parseInt(squadId);
			for(RCONEventHandler handler : handlers) {
				handler.onTeamChange(p, team, squad);
			}
		}
	}
	
	private void onKicked(String name, String reason) {
		Player p = online.getByName(name);
		if(p != null) {
			p.onUpdated();
			for(RCONEventHandler handler : handlers) {
				handler.onKicked(p, reason);
			}
		} else {
			logger.warn("onKicked event for unknown player. " + name + " was kicked for " + reason + ".");
		}
	}
	
	private void onAuthenticated(String name) {
		Player p = online.ensureGetByName(name);
		p.onUpdated();
		for(RCONEventHandler handler : handlers) {
			handler.onAuthenticated(p);
		}
	}
	
	public void addHandler(RCONEventHandler handler) {
		handlers.add(handler);
	}
	
	private void onChat(String name, String text) {
		Player p = online.getByName(name);
		if(p != null) {
			p.onUpdated();
			for(RCONEventHandler handler : handlers) {
				handler.onChat(p, text);
			}
		} else {
			logger.warn("onChat event for unknown player. " + name + " said \"" + text + "\".");
		}
	}
	
	private void onJoin(String name, String eaGuid) {
		if(name.trim().length() > 0) {
			Player p = new Player(name);
			p.setEaGuid(eaGuid);
			online.addPlayer(p);
			p.onUpdated();
			logger.info("Player " + name + " (EAGUID: " + eaGuid + ") joined the game.");
			for(RCONEventHandler handler : handlers) {
				handler.onJoin(p, eaGuid);
			}
		} else {
			logger.warn("Player without a name joined. Ignoring for now.");
		}
	}
	
	private void onKill(String killerName, String killedName, String weapon, boolean headshot) {
		Player killer = online.ensureGetByName(killerName);
		Player killed = online.ensureGetByName(killedName);
		if(killer != null && killed != null) {
			killer.onUpdated();
			killed.onUpdated();
			for(RCONEventHandler handler : handlers) {
				handler.onKill(killer, killed, weapon, headshot);
			}
		} else {
			logger.warn("onKill event for unknown player. " + killerName + "(" + killer + ") killed " + killedName
					+ "(" + killed + ").");
		}
	}
	
	private void onLeave(String name, PlayerInfo info) {
		if(info != null) {
			onPlayerInfo(info);
		}
		Player p = online.ensureGetByName(name);
		p.onUpdated();
		logger.info("Player " + name + " left the game.");
		for(RCONEventHandler handler : handlers) {
			handler.onLeave(p, info);
		}
		online.removePlayer(p);
	}
	
	private void onPunkbuster(String message) {
		Matcher matcher = pListPattern.matcher(message);
		if(matcher.matches()) {
			String guid = matcher.group(2);
			String ip = matcher.group(3);
			String name = matcher.group(5);
			
			Player p = syncPlayer(name, guid, ip);
			for(RCONEventHandler handler : handlers) {
				handler.onPbInfo(p, guid, ip);
			}
		}
		matcher = guidPattern.matcher(message);
		if(matcher.matches()) {
			String guid = matcher.group(1);
			String ip = matcher.group(3);
			String name = matcher.group(5);
			
			Player p = syncPlayer(name, guid, ip);
			for(RCONEventHandler handler : handlers) {
				handler.onPbInfo(p, guid, ip);
			}
		}
		
		for(RCONEventHandler handler : handlers) {
			handler.onPunkbuster(message);
		}
	}
	
	public void onPlayerInfo(PlayerInfo info) {
		Player p = online.ensureGetByName(info.getName());
		p.onUpdated();
		PlayerInfo old = p.getInfo();
		p.update(info);
		for(RCONEventHandler handler : handlers) {
			handler.onPlayerInfo(p, old, info);
		}
	}
	
	private Player syncPlayer(String name, String guid, String ip) {
		Player p = online.ensureGetByName(name);
		p.setPbGuid(guid);
		p.setIp(ip);
		return p;
	}
	
}
