package rcon.server;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.apache.log4j.Logger;

import rcon.Scheduler;
import rcon.commands.ClientRequest;
import rcon.commands.Command;
import rcon.commands.KickCommand;
import rcon.commands.PunkBusterCommand;
import rcon.commands.SayCommand;
import rcon.event.RCONEventHandler;
import rcon.ingame.ServerMessages;
import rcon.map.Map;
import rcon.players.Player;
import rcon.players.PlayerInfo;

public class GameServer {
	
	private static final int		PLAYER_TIMEOUT	= 60000;
	
	static Logger					logger			= Logger.getLogger(GameServer.class);
	
	private final String			host;
	private final int				port;
	private String					serverName;
	private int						maxPlayerCount;
	private String					gameMode;
	private String					map;
	private int						roundsPlayed;
	private int						roundsTotal;
	private long					connectionTime;
	
	private final OnlineList		online;
	private final Configuration		config;
	private ServerMessages			serverMessages;
	
	private final ServerConnection	connection;
	
	private EventParser				events;
	
	private HeartbeatChecker		heartbeats;
	private PlayerUpdater			playerUpdater;
	private ServerMessenger			serverMessenger;
	private ConfigSynchronizer		configSynchronizer;
	
	public GameServer(String host, int port, int timeout, String configFile) throws IOException {
		this.host = host;
		this.port = port;
		this.online = new OnlineList();
		this.config = new Configuration();
		this.config.load(configFile);
		this.connection = new ServerConnection(timeout);
		
		this.heartbeats = null;
		this.events = null;
		this.playerUpdater = null;
		
		this.serverMessenger = null;
		this.serverMessages = new ServerMessages();
		
		this.serverName = "Unknown";
		this.maxPlayerCount = 0;
		this.gameMode = "Unknown";
		this.map = "Unknown";
		this.roundsPlayed = 0;
		this.roundsTotal = 0;
		
		this.configSynchronizer = null;
		
		connection.start(host, port);
		this.connectionTime = -1;
	}
	
	public String getHost() {
		return host;
	}
	
	public int getPort() {
		return port;
	}
	
	public OnlineList getOnline() {
		return online;
	}
	
	public synchronized ServerMessages getServerMessages() {
		return serverMessages;
	}
	
	public synchronized Configuration getConfiguration() {
		return config;
	}
	
	public String getServerName() {
		return serverName;
	}
	
	public int getMaxPlayerCount() {
		return maxPlayerCount;
	}
	
	public String getGameMode() {
		return gameMode;
	}
	
	public String getMap() {
		return map;
	}
	
	public int getRoundsPlayed() {
		return roundsPlayed;
	}
	
	public int getRoundsTotal() {
		return roundsTotal;
	}
	
	public void awaitDisconnect() {
		connection.awaitShutdown();
		if(events != null) {
			try {
				events.join();
			} catch(InterruptedException e) {}
		}
	}
	
	private synchronized void connectionLost() {
		if(events != null) {
			connection.addEvent(Command.EOF);
			try {
				events.join(100);
			} catch(InterruptedException e) {}
			events = null;
		}
		
		if(heartbeats != null) {
			disableHeartbeats();
		}
		if(playerUpdater != null) {
			disablePlayerUpdates();
		}
		if(serverMessenger != null) {
			disableServerMessenger();
		}
		if(configSynchronizer != null) {
			configSynchronizer.stop();
			configSynchronizer = null;
		}
		
		connection.stop();
	}
	
	public synchronized void stop() {
		if(events != null) {
			disableEvents();
		}
		if(heartbeats != null) {
			disableHeartbeats();
		}
		if(playerUpdater != null) {
			disablePlayerUpdates();
		}
		if(serverMessenger != null) {
			disableServerMessenger();
		}
		if(configSynchronizer != null) {
			configSynchronizer.stop();
			configSynchronizer = null;
		}
		
		logout();
		try {
			connection.quit();
		} catch(IOException e) {}
		connection.stop();
	}
	
	public synchronized boolean login(String password) {
		try {
			boolean res = connection.login(password);
			if(res) {
				connectionTime = Calendar.getInstance().getTimeInMillis() / 1000;
				onConnect();
			}
			return res;
		} catch(IOException e) {
			connectionLost();
			return false;
		}
	}
	
	private void onConnect() {
		this.configSynchronizer = new ConfigSynchronizer(this);
		configSynchronizer.run();
		Scheduler.getInstance().repeat(configSynchronizer, 3 * 60 * 1000);
	}
	
	public synchronized boolean enableEvents() {
		if(events != null) {
			throw new IllegalStateException("Event handling alreqady enabled.");
		}
		try {
			Command response = request("admin.eventsEnabled", "true");
			if(response.is("OK")) {
				events = new EventParser(online, connection);
				events.start();
				return true;
			} else {
				return false;
			}
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized void disableEvents() {
		if(events == null) {
			throw new IllegalStateException("Event handling not enabled.");
		}
		try {
			request("admin.eventsEnabled", "false");
		} catch(IOException e) {}
		connection.addEvent(Command.EOF);
		try {
			events.join(500);
		} catch(InterruptedException e) {
			e.printStackTrace();
		}
		events = null;
	}
	
	public synchronized boolean enableHeartbeats(int interval) {
		if(heartbeats != null) {
			throw new IllegalStateException("Heartbeats already started.");
		}
		heartbeats = new HeartbeatChecker(this);
		Scheduler.getInstance().repeat(heartbeats, interval);
		return true;
	}
	
	public synchronized void disableHeartbeats() {
		if(heartbeats == null) {
			throw new IllegalStateException("Heartbeats not started.");
		}
		heartbeats.stop();
		heartbeats = null;
	}
	
	public synchronized boolean enablePlayerUpdates(int interval) {
		if(playerUpdater != null) {
			throw new IllegalStateException("Player updates already enabled");
		}
		playerUpdater = new PlayerUpdater(this, PLAYER_TIMEOUT);
		Scheduler.getInstance().repeat(playerUpdater, interval);
		return true;
	}
	
	public synchronized void disablePlayerUpdates() {
		if(playerUpdater == null) {
			throw new IllegalStateException("Player updates are not enabled.");
		}
		playerUpdater.setEnabled(false);
		playerUpdater = null;
	}
	
	public synchronized void enableServerMessenger(int interval) {
		if(serverMessenger != null) {
			throw new IllegalStateException("Server messenger already enabled.");
		}
		serverMessenger = new ServerMessenger(this, serverMessages);
		Scheduler.getInstance().repeat(serverMessenger, interval);
	}
	
	public synchronized void disableServerMessenger() {
		if(serverMessenger == null) {
			throw new IllegalStateException("Server messenger is not enabled.");
		}
		serverMessenger.stop();
		serverMessenger = null;
	}
	
	public synchronized void logout() {
		try {
			connection.logout();
		} catch(IOException e) {}
	}
	
	public synchronized void addHandler(RCONEventHandler handler) {
		if(events == null) {
			throw new IllegalStateException("Event handling not enabled.");
		}
		events.addHandler(handler);
	}
	
	public synchronized boolean say(String message) {
		try {
			Command response = request(new SayCommand(message));
			return response.is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized boolean say(String message, Player p) {
		try {
			Command response = request(new SayCommand(message, p.getName()));
			return response.is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized boolean disconnect(Player p, String reason) {
		try {
			Command response = request(new KickCommand(p.getName(), reason));
			return response.is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized boolean ban(final Player p, String reason, String duration) {
		String fullDuration = duration.equals("perm") ? "permanently" : ("for a " + duration.toLowerCase());
		
		StringBuilder sb = new StringBuilder();
		sb.append("Player " + p.getName() + " has been banned ");
		sb.append(fullDuration);
		sb.append(" with the reason " + reason + ".");
		say(sb.toString());
		
		sb = new StringBuilder();
		sb.append("You are being banned ");
		sb.append(fullDuration);
		sb.append(" with the reason " + reason + ".");
		say(sb.toString(), p);
		
		sb = new StringBuilder();
		sb.append("Banned ");
		sb.append(fullDuration);
		sb.append(" for " + reason + ".");
		
		final String disconnectMessage = sb.toString();
		Scheduler.getInstance().schedule(new Runnable() {
			@Override
			public void run() {
				disconnect(p, disconnectMessage);
			}
		}, 10000);
		return true;
	}
	
	public synchronized boolean kick(final Player p, final String reason) {
		say("You are being kicked for " + reason + ".", p);
		String text = "Player " + p.getName() + " has been kicked for " + reason + ".";
		say(text);
		
		Scheduler.getInstance().schedule(new Runnable() {
			@Override
			public void run() {
				disconnect(p, reason);
			}
		}, 5000);
		return true;
	}
	
	public synchronized boolean warn(Player p, String reason) {
		String targetMessage = "You were warned for " + reason + ".";
		say(targetMessage, p);
		
		String message;
		message = "Player " + p.getName() + " has been warned for " + reason + ".";
		say(message);
		return true;
	}
	
	public synchronized boolean restartMap() {
		try {
			Command response = request("admin.restartMap");
			return response.is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized boolean runNextMap() {
		try {
			Command response = request("admin.runNextLevel");
			return response.is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized boolean endRound(int winner) {
		try {
			Command response = request("admin.endRound", String.valueOf(winner));
			return response.is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized Map currentMap() {
		try {
			Command response = request("admin.currentLevel");
			if(response.is("OK")) {
				String name = response.get(0);
				Map map = Map.get(name);
				return map;
			} else {
				return null;
			}
		} catch(IOException e) {
			return null;
		}
	}
	
	public synchronized boolean setNextMapIndex(int index) {
		try {
			Command response = request("mapList.nextLevelIndex", String.valueOf(index));
			return response.is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized boolean setPlaylist(String playlist) {
		try {
			Command response = request("admin.setPlaylist", playlist);
			return response.is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized boolean clearMapList() {
		try {
			Command response = request("mapList.clear");
			return response.is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized boolean appendMap(Map map, int rounds) {
		try {
			Command response = request("mapList.append", map.getLevel(), String.valueOf(rounds));
			return response.is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	/*
	 * public synchronized boolean applyConfiguration() {
	 * config.applyConfiguration(this); return true; }
	 */
	
	public synchronized boolean requestPbPlayerList() {
		try {
			Command response = request(new PunkBusterCommand("pb_sv_Plist"));
			return response.is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized int getVersion() {
		try {
			Command response = request("version");
			if(response.is("OK")) {
				try {
					return response.getInt(1);
				} catch(Exception e) {
					return -1;
				}
			} else {
				return -1;
			}
		} catch(IOException e) {
			return -1;
		}
	}
	
	public synchronized boolean requestPlayerInfo() {
		try {
			Command response = request("admin.listPlayers", "all");
			if(response.is("OK")) {
				ArrayList<PlayerInfo> playerInfo = PlayerInfo.parse(response);
				if(playerInfo.size() != online.size()) {
					logger.warn("Actual player count " + playerInfo.size() + " does not match local player count "
							+ online.size() + ".");
				}
				for(PlayerInfo info : playerInfo) {
					updatePlayer(info);
				}
				return true;
			}
			return false;
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized boolean updateInfo() {
		try {
			Command response = request("serverInfo");
			if(response.is("OK")) {
				serverName = response.get(0);
				maxPlayerCount = response.getInt(2);
				gameMode = response.get(3);
				map = response.get(4);
				roundsPlayed = response.getInt(5);
				roundsTotal = response.getInt(6);
				// int teamCount = response.getInt(7);
				// int team1Score = response.getInt(7+1);
				// int team2Score = response.getInt(7+2);
				// etc. ?
				// OK Timeless Rush II - www.TimelessGaming.com 0 32 RUSH
				// Levels/MP_012GR 1 2 2 100 1500 0
				return true;
			} else {
				return false;
			}
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized boolean kill(Player target) {
		try {
			Command response = request("admin.killPlayer", target.getName());
			return response.is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized boolean setVariable(String variable, String value) {
		try {
			return request(variable, value).is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized String getVariable(String variable) {
		try {
			Command response = request(variable);
			if(response.is("OK") && response.size() > 0) {
				return response.get(0);
			} else {
				return null;
			}
		} catch(IOException e) {
			return null;
		}
	}
	
	public synchronized boolean punkbuster(String command) {
		try {
			Command response = request("punkBuster.pb_sv_command", command);
			return response.is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized List<String> getReservedSlots() {
		try {
			Command response = request("reservedSlots.list");
			if(response.is("OK")) {
				ArrayList<String> slots = new ArrayList<String>();
				for(int i = 0; i < response.size(); i++) {
					slots.add(response.get(i));
				}
				return slots;
			} else {
				return null;
			}
		} catch(IOException e) {
			return null;
		}
	}
	
	public synchronized boolean addReservedSlot(String playerName) {
		try {
			return request("reservedSlots.addPlayer", playerName).is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized boolean removeReservedSlot(String playerName) {
		try {
			return request("reservedSlots.removePlayer", playerName).is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized boolean clearReservedSlots() {
		try {
			return request("reservedSlots.clear").is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized boolean loadReservedSlots() {
		try {
			return request("reservedSlots.load").is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	public synchronized boolean saveReservedSlots() {
		try {
			return request("reservedSlots.save").is("OK");
		} catch(IOException e) {
			return false;
		}
	}
	
	private synchronized void updatePlayer(PlayerInfo info) {
		Player player = online.ensureGetByName(info.getName());
		if(events != null) {
			events.onPlayerInfo(info);
		}
	}
	
	private synchronized Command request(String command, String... arguments) throws IOException {
		return request(new ClientRequest(command, arguments));
	}
	
	private synchronized Command request(Command command) throws IOException {
		try {
			return connection.sendRequest(command);
		} catch(IOException e) {
			connectionLost();
			throw e;
		}
	}
	
	public long getConnectionTime() {
		return connectionTime;
	}
	
}
