package rcon.server;

import java.util.ArrayList;
import java.util.Arrays;

public class SimpleTimer {
	
	private volatile ArrayList<Runnable>	listeners;
	private volatile int					delay;
	private volatile long					last;
	private volatile boolean				started;
	private volatile boolean				closed;
	private volatile Object					lock;
	private volatile TimerThread			thread;
	
	public SimpleTimer(int delay, Runnable... l) {
		this.listeners = new ArrayList<Runnable>(Arrays.asList(l));
		this.delay = delay;
		this.last = 0;
		this.started = false;
		this.closed = false;
		this.lock = new Object();
		this.thread = new TimerThread();
		this.thread.start();
	}
	
	public void close() {
		synchronized(lock) {
			started = false;
			closed = true;
			lock.notifyAll();
		}
		try {
			thread.join(100);
		} catch(InterruptedException e) {}
	}
	
	public void start() {
		synchronized(lock) {
			started = true;
			last = System.currentTimeMillis();
			lock.notifyAll();
		}
	}
	
	public void restart() {
		start();
	}
	
	public void stop() {
		synchronized(lock) {
			started = false;
			lock.notifyAll();
		}
	}
	
	public void abort() {
		stop();
	}
	
	private class TimerThread extends Thread {
		
		public TimerThread() {
			super("TimerThread");
			this.setDaemon(true);
		}
		
		@Override
		public void run() {
			synchronized(lock) {
				while(!closed) {
					while(!closed && (!started || System.currentTimeMillis() < last + delay)) {
						try {
							if(started) {
								lock.wait(Math.max(0, delay - System.currentTimeMillis() + last));
							} else {
								lock.wait();
							}
						} catch(InterruptedException e) {
							e.printStackTrace();
						}
					}
					if(closed) {
						break;
					}
					for(Runnable l : listeners) {
						l.run();
					}
					last = System.currentTimeMillis();
					started = false;
					lock.notifyAll();
				}
			}
		}
		
	}
	
}
