/*
 * Decompiled with CFR 0.152.
 */
package game.entities;

import game.engine.BaseEntity;
import game.engine.BoundingBox;
import game.engine.loader.EntityDescriptor;
import game.engine.state.AbstractState;
import game.engine.state.State;
import game.engine.state.StateMachine;
import game.entities.Character;
import game.entities.Entity;
import game.entities.Player;
import game.entities.SpeechBubble;
import game.entities.Waypoint;
import game.world.GameWorld;
import game.world.MapGameWorld;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Random;
import org.newdawn.slick.Animation;
import org.newdawn.slick.Color;
import org.newdawn.slick.GameContainer;
import org.newdawn.slick.Graphics;
import org.newdawn.slick.geom.Vector2f;

public class Enemy
extends Character {
    private static final float NOTIFY_DISTANCE = 250.0f;
    private Direction direction = Direction.left;
    private int interactionCooldown;
    private float detectRadiusOuter;
    private float detectRadiusInner;
    private boolean canRecognizePlayer;
    private StateMachine<EnemyState, State> states;

    public Enemy(EntityDescriptor descriptor) {
        super(descriptor);
        this.enableCustomRendering();
        this.interactionCooldown = 0;
        this.states = new StateMachine(EnemyState.class);
        this.states.put(EnemyState.idling, new IdleState());
        this.states.put(EnemyState.detecting, new DetectingState());
        this.states.put(EnemyState.pursuing, new PursuingState());
        this.states.put(EnemyState.interacting, new InteractingState());
        this.states.put(EnemyState.returning, new ReturningState());
        this.states.put(EnemyState.lost, new LostState());
        this.states.put(EnemyState.knockback, new KnockbackState());
        this.detectRadiusInner = this.properties.getFloat("detect_radius_inner", 250.0f);
        this.detectRadiusOuter = this.properties.getFloat("detect_radius_outer", 500.0f);
        this.canRecognizePlayer = this.properties.getBoolean("detect_player", true);
    }

    public void update(int dt, GameWorld world, GameContainer container) {
        super.update(dt, world, container);
        this.interactionCooldown -= dt;
        if (this.states.is(null)) {
            this.states.transition(world, EnemyState.idling);
        }
        this.states.updateCurrentState(dt, world, container);
    }

    private float getSquaredDistanceTo(BaseEntity entity) {
        Vector2f p = this.getPos();
        BoundingBox b = this.getBounds();
        float dx = b.getCenterX(p);
        float dy = b.getCenterY(p);
        p = entity.getPos();
        b = entity.getBounds();
        float distSquared = (dx -= b.getCenterX(p)) * dx + (dy -= b.getCenterY(p)) * dy;
        return distSquared;
    }

    private float move(float distance, Vector2f towards) {
        Vector2f temp = new Vector2f();
        Vector2f pos = this.getPos();
        temp.set(pos);
        temp.sub(towards);
        float remainingDistance = temp.length();
        if (remainingDistance > distance) {
            temp.scale(distance / remainingDistance);
            distance = 0.0f;
        } else {
            distance -= remainingDistance;
        }
        this.translate(-temp.x, -temp.y);
        this.direction = temp.x > 0.0f ? Direction.left : Direction.right;
        return distance;
    }

    private Vector2f[] findPath(MapGameWorld world, Vector2f to) {
        Vector2f pos = this.getPos();
        BoundingBox collisionBox = this.getCollisionBox();
        Vector2f from = new Vector2f(collisionBox.getCenterX(pos), collisionBox.getCenterY(pos));
        Vector2f[] path = world.getPathSystem().findShotestPath(from, to);
        if (path != null) {
            for (Vector2f v : path) {
                v.x -= collisionBox.getCenterX(0.0f, 0.0f);
                v.y -= collisionBox.getCenterY(0.0f, 0.0f);
            }
        }
        return path;
    }

    private int moveAlongPath(float distance, Vector2f[] path, int nextPathIndex) {
        Vector2f next = new Vector2f();
        while (distance > 0.0f && nextPathIndex < path.length) {
            next.set(path[nextPathIndex].x, path[nextPathIndex].y);
            if (!((distance = this.move(distance, next)) > 0.0f)) continue;
            ++nextPathIndex;
        }
        return nextPathIndex;
    }

    public void render(Graphics g, GameWorld world, GameContainer container) {
        String animationState;
        super.render(g, world, container);
        switch (this.states.getCurrentState()) {
            case idling: 
            case pursuing: 
            case returning: {
                animationState = "walk";
                break;
            }
            case interacting: {
                animationState = "shake";
                break;
            }
            case lost: 
            case detecting: 
            case knockback: {
                animationState = "idle";
                break;
            }
            default: {
                throw new RuntimeException("Don't know which animation state to use when " + this.states.getCurrentState().toString() + ".");
            }
        }
        Animation animation = (Animation)this.animations.get(animationState);
        g.pushTransform();
        g.scale(this.direction.scale, 1.0f);
        float x = 0.5f * (this.direction.scale - 1.0f) * (float)animation.getWidth();
        g.drawAnimation(animation, x, 0.0f);
        g.popTransform();
    }

    public void debugRender(Graphics g, GameWorld world, GameContainer container) {
        super.debugRender(g, world, container);
        Vector2f pos = this.getPos();
        BoundingBox bounds = this.getCollisionBox();
        if (bounds != null) {
            g.setColor(new Color(1.0f, 0.0f, 0.0f, 0.3f));
            g.fillRect(bounds.getMinX(pos), bounds.getMinY(pos), bounds.getWidth(), bounds.getHeight());
            g.setColor(Color.red);
            g.drawRect(bounds.getMinX(pos), bounds.getMinY(pos), bounds.getWidth(), bounds.getHeight());
        }
        switch (this.states.getCurrentState()) {
            case idling: 
            case detecting: {
                this.debugRenderIdleState(g);
                break;
            }
            case returning: {
                this.debugRenderReturningState(g);
            }
            case pursuing: {
                this.debugRenderPursuingState(g);
                break;
            }
        }
    }

    private void debugRenderReturningState(Graphics g) {
        ReturningState returningState = (ReturningState)this.states.getStateObject(EnemyState.returning);
        Vector2f[] path = returningState.path;
        if (path == null) {
            return;
        }
        this.renderPath(g, path, returningState.nextPathIndex);
    }

    private void debugRenderPursuingState(Graphics g) {
        PursuingState pursuingState = (PursuingState)this.states.getStateObject(EnemyState.pursuing);
        Vector2f[] path = pursuingState.path;
        if (path == null) {
            return;
        }
        this.renderPath(g, path, pursuingState.nextPathIndex);
    }

    private void renderPath(Graphics g, Vector2f[] path, int nextIndex) {
        BoundingBox collisionBox = this.getCollisionBox();
        float ox = collisionBox.getCenterX(0.0f, 0.0f);
        float oy = collisionBox.getCenterY(0.0f, 0.0f);
        Vector2f from = new Vector2f();
        Vector2f to = new Vector2f();
        to.set(this.getPos());
        to.x += ox;
        to.y += oy;
        g.setColor(Color.green);
        for (int i = nextIndex; i < path.length; ++i) {
            from.set(to);
            to.x = path[i].x + ox;
            to.y = path[i].y + oy;
            g.drawLine(from.x, from.y, to.x, to.y);
            g.fillOval(from.x - 2.0f, from.y - 2.0f, 4.0f, 4.0f);
        }
    }

    private void debugRenderIdleState(Graphics g) {
        IdleState idleState = (IdleState)this.states.getStateObject(EnemyState.idling);
        BoundingBox collisionBox = this.getCollisionBox();
        float ox = collisionBox.getCenterX(0.0f, 0.0f);
        float oy = collisionBox.getCenterY(0.0f, 0.0f);
        Vector2f from = idleState.previousWaypoint.getCenter();
        Vector2f to = new Vector2f(idleState.targetWaypointPosition);
        Vector2f current = new Vector2f(this.getPos());
        to.x += ox;
        to.y += oy;
        current.x += ox;
        current.y += oy;
        g.setColor(Color.green);
        g.drawLine(current.x, current.y, to.x, to.y);
        g.fillOval(current.x - 5.0f, current.y - 5.0f, 10.0f, 10.0f);
        g.setColor(Color.orange);
        g.drawLine(from.x, from.y, current.x, current.y);
    }

    public void setWaypoints(Waypoint from, Waypoint to, Random random) {
        IdleState idleState = (IdleState)this.states.getStateObject(EnemyState.idling);
        idleState.previousWaypoint = from;
        idleState.nextWaypoint = to;
        idleState.nextWaypoint.getCenter(idleState.targetWaypointPosition);
        BoundingBox collisionBox = this.getCollisionBox();
        ((IdleState)idleState).targetWaypointPosition.x += random.nextFloat() * 50.0f - 25.0f - collisionBox.getCenterX(0.0f, 0.0f);
        ((IdleState)idleState).targetWaypointPosition.y += random.nextFloat() * 50.0f - 25.0f - collisionBox.getCenterY(0.0f, 0.0f);
    }

    protected void getDebugText(StringBuilder sb) {
        super.getDebugText(sb);
        sb.append("State: " + this.states.getCurrentState().toString() + "\n");
    }

    public void onStopInteracting(GameWorld world) {
        if (this.states.is(EnemyState.interacting)) {
            this.states.transition(world, EnemyState.returning);
        }
    }

    public void knockBack(GameWorld world) {
        if (!this.states.is(EnemyState.idling)) {
            this.states.transition(world, EnemyState.knockback);
        }
    }

    private class KnockbackState
    extends AbstractState {
        private static final float KNOCKBACK_DISTANCE = 100.0f;
        private static final float KNOCKBACK_SPEED = 500.0f;
        private static final int KNOCKBACK_DURATION = 200;
        private static final int KNOCKOUT_DURATION = 500;
        private final Vector2f direction = new Vector2f();
        private int time = 0;

        public void enter(GameWorld world) {
            this.time = 0;
            Player player = world.getEntities().getOne(Player.class);
            if (player == null) {
                Enemy.this.states.transition(world, Enemy.this.interactionCooldown <= 0 ? EnemyState.pursuing : EnemyState.returning);
            } else {
                this.direction.set(Enemy.this.getPos());
                this.direction.sub(player.getPos());
                this.direction.normalise();
            }
        }

        public void update(int dt, GameWorld world, GameContainer container) {
            this.time += dt;
            if (this.time < 200) {
                Enemy.this.translate(this.direction.x * 500.0f * (float)dt / 1000.0f, this.direction.y * 500.0f * (float)dt / 1000.0f);
            }
            if (this.time >= 700) {
                Enemy.this.states.transition(world, Enemy.this.interactionCooldown <= 0 ? EnemyState.pursuing : EnemyState.returning);
            }
        }
    }

    private class LostState
    extends AbstractState {
        private final int lostDuration;
        private int lostWarmup;

        public LostState() {
            this.lostDuration = Enemy.this.properties.getInt("lost_duration", 1000);
            this.lostWarmup = 0;
        }

        public void enter(GameWorld world) {
            this.lostWarmup = 0;
        }

        public void update(int dt, GameWorld world, GameContainer container) {
            float distSquared;
            this.lostWarmup += dt;
            Player player = world.getEntities().getOne(Player.class);
            if (player != null && this.lostWarmup > this.lostDuration && (distSquared = Enemy.this.getSquaredDistanceTo(player)) < Enemy.this.detectRadiusInner * Enemy.this.detectRadiusInner) {
                Enemy.this.states.transition(world, EnemyState.pursuing);
            }
        }
    }

    private class ReturningState
    extends AbstractState {
        private final float walkSpeed;
        private Vector2f[] path;
        private int nextPathIndex;

        public ReturningState() {
            this.walkSpeed = Enemy.this.properties.getFloat("walk_speed", 100.0f);
            this.path = null;
            this.nextPathIndex = 0;
        }

        public void enter(GameWorld world) {
            IdleState idleState = (IdleState)Enemy.this.states.getStateObject(EnemyState.idling);
            this.path = Enemy.this.findPath((MapGameWorld)world, idleState.nextWaypoint.getCenter());
            this.nextPathIndex = 0;
            if (this.path == null) {
                this.findFallbackWaypoint(world);
            }
            if (this.path == null) {
                Enemy.this.states.transition(world, EnemyState.lost);
            }
        }

        public void update(int dt, GameWorld world, GameContainer container) {
            if (this.nextPathIndex < this.path.length) {
                float movedDistance = (float)dt * this.walkSpeed / 1000.0f;
                this.nextPathIndex = Enemy.this.moveAlongPath(movedDistance, this.path, this.nextPathIndex);
                Player player = world.getEntities().getOne(Player.class);
                float distSquared = Enemy.this.getSquaredDistanceTo(player);
                if (distSquared < Enemy.this.detectRadiusInner * Enemy.this.detectRadiusInner && Enemy.this.interactionCooldown <= 0) {
                    Enemy.this.states.transition(world, EnemyState.pursuing);
                }
            } else {
                Enemy.this.states.transition(world, EnemyState.idling);
            }
        }

        private void findFallbackWaypoint(GameWorld world) {
            final Vector2f wp = new Vector2f();
            final Vector2f mp = new Vector2f();
            BoundingBox collisionBox = Enemy.this.getCollisionBox();
            mp.x = collisionBox.getCenterX(Enemy.this.getPos());
            mp.y = collisionBox.getCenterY(Enemy.this.getPos());
            ArrayList<Waypoint> waypoints = new ArrayList<Waypoint>((Collection<Waypoint>)((Object)world.getEntities().get((Entity)((Object)Waypoint.class))));
            Collections.sort(waypoints, new Comparator<Waypoint>(){

                @Override
                public int compare(Waypoint o1, Waypoint o2) {
                    o1.getCenter(wp);
                    float a = mp.distanceSquared(wp);
                    o2.getCenter(wp);
                    float b = mp.distanceSquared(wp);
                    if (a < b) {
                        return -1;
                    }
                    if (a > b) {
                        return 1;
                    }
                    return 0;
                }
            });
            for (int i = 0; i < Math.min(5, waypoints.size()); ++i) {
                Waypoint w = waypoints.get(i);
                this.path = Enemy.this.findPath((MapGameWorld)world, w.getCenter());
                this.nextPathIndex = 0;
                if (this.path == null) continue;
                Enemy.this.setWaypoints(w, w, world.getRandom());
                break;
            }
        }
    }

    private class PursuingState
    extends AbstractState {
        private static final int PATH_LIFE_MIN = 200;
        private static final int PATH_LIFE_MAX = 600;
        private static final int BROADCAST_COOLDOWN_MAX = 1500;
        private static final int BROADCAST_COOLDOWN_MIN = 500;
        private final float interactDistance;
        private final float runSpeed;
        private Vector2f[] path;
        private int nextPathIndex;
        private int pathLifeTime;
        private int broadcastCooldown;

        public PursuingState() {
            this.interactDistance = Enemy.this.properties.getFloat("interact_distance", 50.0f);
            this.runSpeed = Enemy.this.properties.getFloat("run_speed", 350.0f);
            this.path = null;
            this.nextPathIndex = 0;
            this.pathLifeTime = 0;
            this.broadcastCooldown = 0;
        }

        public void enter(GameWorld world) {
            this.path = null;
            this.nextPathIndex = 0;
            this.pathLifeTime = 0;
            this.broadcastCooldown = this.getRandomCooldown(world);
        }

        public void update(int dt, GameWorld world, GameContainer container) {
            this.pathLifeTime -= dt;
            this.broadcastCooldown -= dt;
            Player player = world.getEntities().getOne(Player.class);
            if (player != null) {
                float distSquared;
                if (this.path == null || this.pathLifeTime <= 0 || this.nextPathIndex == this.path.length) {
                    Vector2f target = new Vector2f();
                    target.x = player.getCollisionBox().getCenterX(player.getPos());
                    target.y = player.getCollisionBox().getCenterY(player.getPos());
                    Vector2f[] path = Enemy.this.findPath((MapGameWorld)world, target);
                    if (path != null) {
                        this.nextPathIndex = 0;
                        this.pathLifeTime = world.getRandom().nextInt(400) + 200;
                        this.path = path;
                    }
                }
                if ((distSquared = Enemy.this.getSquaredDistanceTo(player)) < this.interactDistance * this.interactDistance) {
                    Enemy.this.states.transition(world, EnemyState.interacting);
                } else if (distSquared > Enemy.this.detectRadiusOuter * Enemy.this.detectRadiusOuter) {
                    Enemy.this.states.transition(world, EnemyState.returning);
                } else if (this.path != null && this.nextPathIndex < this.path.length) {
                    float movedDistance = (float)dt * this.runSpeed / 1000.0f;
                    this.nextPathIndex = Enemy.this.moveAlongPath(movedDistance, this.path, this.nextPathIndex);
                    if (this.broadcastCooldown <= 0 && Enemy.this.canRecognizePlayer) {
                        Iterator i$ = world.getEntities().get((Entity)((Object)Enemy.class)).iterator();
                        while (i$.hasNext()) {
                            Enemy e = (Enemy)i$.next();
                            if (e == Enemy.this || !(e.getPos().distanceSquared(Enemy.this.getPos()) <= 62500.0f) || !e.states.is(EnemyState.idling)) continue;
                            e.states.transition(world, EnemyState.detecting);
                        }
                        SpeechBubble bubble = (SpeechBubble)world.getLoader().create("speech_bubble");
                        bubble.positionAbove(Enemy.this);
                        world.getEntities().add(bubble);
                        this.broadcastCooldown = this.getRandomCooldown(world);
                    }
                }
            }
        }

        private int getRandomCooldown(GameWorld world) {
            return world.getRandom().nextInt(1000) + 500;
        }
    }

    private class InteractingState
    extends AbstractState {
        private final int interactDuration;
        private final int interactCooldown;
        private int interactionTime = 0;

        public InteractingState() {
            this.interactDuration = Enemy.this.properties.getInt("interact_duration", 2000);
            this.interactCooldown = Enemy.this.properties.getInt("interact_cooldown", 10000);
        }

        public void enter(GameWorld world) {
            this.interactionTime = 0;
            Player player = world.getEntities().getOne(Player.class);
            if (player != null) {
                player.onStartInteracting(world, Enemy.this);
            }
        }

        public void leave(GameWorld world) {
            Player player = world.getEntities().getOne(Player.class);
            if (player != null) {
                player.onStopInteracting(world, Enemy.this);
            }
        }

        public void update(int dt, GameWorld world, GameContainer container) {
            this.interactionTime += dt;
            if (this.interactionTime > this.interactDuration) {
                Enemy.this.states.transition(world, EnemyState.returning);
                Enemy.this.interactionCooldown = this.interactCooldown;
            }
        }
    }

    private class DetectingState
    extends AbstractState {
        private int detectTime = 0;
        private final int detectDurationPre;
        private final int detectDurationPost;

        public DetectingState() {
            this.detectDurationPre = Enemy.this.properties.getInt("detect_duration_pre", 500);
            this.detectDurationPost = Enemy.this.properties.getInt("detect_duration_post", 500);
        }

        public void enter(GameWorld world) {
            this.detectTime = 0;
        }

        public void update(int dt, GameWorld world, GameContainer container) {
            Player player = world.getEntities().getOne(Player.class);
            if (player != null) {
                float distSquared = Enemy.this.getSquaredDistanceTo(player);
                if (distSquared > Enemy.this.detectRadiusOuter * Enemy.this.detectRadiusOuter || Enemy.this.interactionCooldown > 0) {
                    Enemy.this.states.transition(world, EnemyState.idling);
                } else {
                    if (this.detectTime <= this.detectDurationPre && this.detectTime + dt > this.detectDurationPre && Enemy.this.canRecognizePlayer) {
                        SpeechBubble bubble = (SpeechBubble)world.getLoader().create("speech_bubble");
                        bubble.positionAbove(Enemy.this);
                        world.getEntities().add(bubble);
                        Iterator i$ = world.getEntities().get((Entity)((Object)Enemy.class)).iterator();
                        while (i$.hasNext()) {
                            Enemy e = (Enemy)i$.next();
                            if (e == Enemy.this || !(e.getPos().distanceSquared(Enemy.this.getPos()) <= 62500.0f) || !e.states.is(EnemyState.idling)) continue;
                            e.states.transition(world, EnemyState.detecting);
                        }
                    }
                    this.detectTime += dt;
                    if (this.detectTime > this.detectDurationPre + this.detectDurationPost) {
                        Enemy.this.states.transition(world, EnemyState.pursuing);
                    }
                }
            }
        }
    }

    private class IdleState
    extends AbstractState {
        private Waypoint previousWaypoint;
        private Waypoint nextWaypoint;
        private final Vector2f targetWaypointPosition = new Vector2f();
        private final float walkSpeed;

        public IdleState() {
            this.walkSpeed = Enemy.this.properties.getFloat("walk_speed", 100.0f);
        }

        public void update(int dt, GameWorld world, GameContainer container) {
            float distSquared;
            float movedDistance = (float)dt * this.walkSpeed / 1000.0f;
            while (movedDistance > 0.0f) {
                if (!((movedDistance = Enemy.this.move(movedDistance, this.targetWaypointPosition)) > 0.0f)) continue;
                Waypoint next = this.nextWaypoint.pickRandomWaypoint(world.getRandom());
                Enemy.this.setWaypoints(this.nextWaypoint, next, world.getRandom());
            }
            Player player = world.getEntities().getOne(Player.class);
            if (player != null && Enemy.this.interactionCooldown <= 0 && (Enemy.this.canRecognizePlayer || player.isHazmatMode()) && (distSquared = Enemy.this.getSquaredDistanceTo(player)) < Enemy.this.detectRadiusInner * Enemy.this.detectRadiusInner) {
                Enemy.this.states.transition(world, EnemyState.detecting);
            }
        }
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    private static enum Direction {
        left(-1.0f),
        right(1.0f);

        public final float scale;

        private Direction(float scale) {
            this.scale = scale;
        }
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    private static enum EnemyState {
        idling,
        detecting,
        pursuing,
        interacting,
        returning,
        lost,
        knockback;

    }
}

